package kswriter

import (
	"context"
	"errors"
	"time"

	v1 "code.justin.tv/amzn/streamlogclient/data/v1"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kinesis"
	"github.com/golang/protobuf/proto"
)

// ErrQueueFull indicates that a message was dropped because the Streamlog client's
// message queue is full.
var ErrQueueFull = errors.New("streamlog queue full, message not sent")

// StartEventKinesisQ for setting eventKinesisQueue
func newEventKinesisQ(kc minimalKinesis, cfg Settings, eh errorHandler, logger logger, stats Statter) *eventKinesisQ {
	var td time.Duration
	ekq := &eventKinesisQ{
		ct:                   0,
		q:                    make(chan *v1.Event, cfg.QueueSize),
		recs:                 getNewPutRecordsInput(cfg.KinesisStreamName),
		kc:                   kc,
		tick:                 time.NewTimer(td),
		kinesisStreamName:    cfg.KinesisStreamName,
		drainEventQueueTimer: cfg.DrainEventQueueTimer,
		errHandeler:          eh,
		logger:               logger,
		stats:                stats,
	}
	return ekq
}

// errorHandler is the interface to error buffer
type errorHandler interface {
	AddError(err error, kr kinesis.PutRecordsInput)
}

// kinesis q for events
// about - the event kinesis queue is a struct that holds:
// - ct: the byte counter for sending events to kinesis whenever thereshold is reached
//   based on the response from chkBillCount
// - q: the channel that EventRecords on sent on
// - recs: structure that holds the actual data to be sent
//
// - to do:
//    -- implement timer based method as well (in conjunction with count) in run method
//	  -- if needed implement a kinesis pool if garbage generation from creating a new
//       kinesis client impacts performance under load
//
type eventKinesisQ struct {
	ct                   int
	q                    chan *v1.Event
	recs                 kinesis.PutRecordsInput
	kc                   minimalKinesis
	tick                 *time.Timer
	kinesisStreamName    string
	drainEventQueueTimer time.Duration
	errHandeler          errorHandler
	logger               logger
	stats                Statter
}

// AddEvent adds the event to the event queue without blocking and returns error if the queue is full.
func (k *eventKinesisQ) AddEvent(e *v1.Event) error {
	select {
	case k.q <- e:
		return nil
	default:
		return ErrQueueFull
	}
}

// eventKinesisQ.run just listens on the q channel for new Recs
func (k *eventKinesisQ) run(ctx context.Context) {
	defer close(k.q)
	defer k.tick.Stop()
	var e *v1.Event
	for {
		select {
		case <-ctx.Done():
			return
		case e = <-k.q:
			k.tick.Stop()
			k.proc(ctx, e)
			k.setTick()
		case <-k.tick.C:
			k.logger.Trace("streamlog.eventKinesisQ.run", "flush timer returned")
			r := k.recs
			if len(r.Records) != 0 {
				k.ct = 0
				k.recs = getNewPutRecordsInput(k.kinesisStreamName)
				go k.batch(ctx, r)
			}
			k.newTick()
		}
	}
}

// eventKinesisQ.proc take a *EventRecs and performs the following:
// - gets bytes from the EventRec to add to the data to be sent to kinesis
//   along with the appropriate partition key (using the UID)
// - increments the queue counter
// - uses chkBillCount to see if eventKinesisQ.batch should be called^
func (k *eventKinesisQ) proc(ctx context.Context, e *v1.Event) {
	if e.Channel == "" {
		k.logger.Error(
			"streamlog.eventKinesisQ.proc",
			"got event without channel. event: %s",
			e.String(),
		)
		return
	}
	b, err := proto.Marshal(e)
	if err != nil {
		k.logger.Error(
			"streamlog.eventKinesisQ.proc",
			"got err marshalling protobuff: %v",
			err,
		)
		return
	}
	k.recs.Records = append(
		k.recs.Records,
		&kinesis.PutRecordsRequestEntry{
			Data:         b,
			PartitionKey: &e.Channel,
		},
	)

	// Amazon bills PutRecord requests as data itself + partition key
	k.ct = k.ct + len(b) + len(e.Channel)
	if chkBillCount(k.ct) || len(k.recs.Records) >= 500 {
		k.stats.Count("streamlog.kinesis.bytes", int64(k.ct))
		// copy the records to make this non blocking
		r := k.recs
		k.ct = 0
		k.recs = getNewPutRecordsInput(k.kinesisStreamName)
		go k.batch(ctx, r)
	}
}

// actually send the kinesis.RequestArgs to kinesis and then reset the
// appropriate fields in the eventKinesisQ
func (k *eventKinesisQ) batch(ctx context.Context, recs kinesis.PutRecordsInput) {
	errorMap := make(map[string]int64)
	k.stats.Count("streamlog.kinesis.put", 1)
	resp, err := k.kc.PutRecordsWithContext(ctx, &recs)
	k.logger.Trace("streamlog.eventKinesisQ.batch", "sent to kinesis")
	if err != nil || resp == nil {
		k.stats.Count("streamlog.kinesis.fail", 1)
		k.logger.Error("streamlog.eventKinesisQ.batch", "got err trying to send to AWS: %s", err)
		k.errHandeler.AddError(err, recs)
		return
	}
	k.stats.Count("streamlog.kinesis.failedrecordct", aws.Int64Value(resp.FailedRecordCount))
	if aws.Int64Value(resp.FailedRecordCount) == 0 {
		return
	}
	for i, rec := range resp.Records {
		select {
		default:
			if aws.StringValue(rec.ErrorCode) == "" {
				continue
			}
			errorMap[aws.StringValue(rec.ErrorCode)]++
			if i < 100 {
				k.logger.Debug("streamlog.eventKinesisQ.batch", "publish.eventq.error.%s: %s", aws.StringValue(rec.ErrorCode), aws.StringValue(rec.ErrorMessage))
			}
		}
	}
	for ky, v := range errorMap {
		select {
		default:
			k.stats.Count("streamlog.kinesis.failedrecord."+ky, v)
			k.logger.Debug("streamlog.eventKinesisQ.batch", "publish.eventq.failedrecord.%s: %d", ky, v)
		}
	}
}

// newTick replaces eventKinesisQ.tick with a new timer
func (k *eventKinesisQ) newTick() {
	k.tick = time.NewTimer(k.drainEventQueueTimer)
}

// setTick resets eventKinesisQ.tick with the appropriate time
func (k *eventKinesisQ) setTick() {
	k.tick.Reset(k.drainEventQueueTimer)
}

// convenience func for getting new kinesis.PutRecordsInput
func getNewPutRecordsInput(streamName string) kinesis.PutRecordsInput {
	return kinesis.PutRecordsInput{
		// allocating a slice of length 5 ... is there a scenario where
		// k.ct could reach maximum before 5 records were entered?
		// could just allocate to 0 and not worry about the performance
		// impact of append calls
		Records:    make([]*kinesis.PutRecordsRequestEntry, 0, 5),
		StreamName: &streamName,
		//StreamName: *flagKinesisStreamName,
	}
}

// chkBillCount is used to check the event counter from the event queue to
// see if it should be batched
// right now checks against a hard coded number ... should this be configurable?
// is the strategy here the right one?
func chkBillCount(i int) bool {
	return i >= 4750
}
