package kswriter

import (
	"fmt"
	"time"

	v1 "code.justin.tv/amzn/streamlogclient/data/v1"
)

// eventHandler is the interface to the event kinesis queue
type eventHandler interface {
	AddEvent(event *v1.Event) error
}

// KinesisStreamlogWriter implements the Streamlog interface and will write events directly to kinesis.
type KinesisStreamlogWriter struct {
	sessionID  string
	serviceTag string
	logger     logger
	eventQ     eventHandler
	killSwitch func()
}

// SetSessionID updates the default streamlog client's Session ID for all future messages.
// This is required for Delay streams, for example, as their Session IDs are different
// from their original stream.
func (sl *KinesisStreamlogWriter) SetSessionID(sessionID string) {
	sl.sessionID = sessionID
}

// Log queues a message to send to streamlog. This method returns immediately, regardless
// of whether the message was sent or queued.
// This method will panic if the client's connection has been closed via Close().
func (sl *KinesisStreamlogWriter) Log(channel, key string, value interface{}) error {
	return sl.LogWithSessionId("", channel, key, value)
}

// LogWithSessionId is like Log, but it uses the provided Session ID instead of the one
// the client was configured with.
// If the `sessionId` argument is the empty string, the client's Session ID is used;
// `LogWithSessionId("", channel, key, value)` is equivalent to `Log(channel, key, value)`
func (sl *KinesisStreamlogWriter) LogWithSessionId(sessionId, channel, key string, value interface{}) error {
	record := &v1.Event_Record{Timestamp: time.Now().Unix()}

	// Fill record data depending on data type
	switch v := value.(type) {
	case int:
		record.I = int64(v)
	case int64:
		record.I = v
	case float32:
		record.F = float64(v)
	case float64:
		record.F = v
	case string:
		record.S = v
	case bool:
		record.B = v
	default:
		return fmt.Errorf("'%T' is not a supported streamlog type", value)
	}

	event := &v1.Event{
		Channel:    channel,
		SessionID:  sl.getSessionId(sessionId),
		ServiceTag: sl.serviceTag,
		Metric:     key,
		Records:    []*v1.Event_Record{record},
	}

	if err := sl.eventQ.AddEvent(event); err != nil {
		sl.logger.Warn("streamlog.LogWithSessionId", "%s: %+v", err, event)
		return err
	}
	return nil

}

// Close the connection to streamlog. This will shut down all helper goroutines.
// Attempts to log more events will panic.
func (sl *KinesisStreamlogWriter) Close() {
	sl.killSwitch()
}

func (sl *KinesisStreamlogWriter) getSessionId(sessionId string) string {
	if sessionId != "" {
		return sessionId
	}
	return sl.sessionID
}
