package kswriter

import (
	"context"
	"errors"
	"testing"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

// kinesis provides an interface to the AWS Kinesis library allowing for the
// simulation of interaction with Kinesis while testing.
//
// All methods will return nil, $error
//

var (
	notImplementedError = errors.New("kinesis: method not implemented")
)

type mockKinesis struct {
}

func (k *mockKinesis) DescribeStreamWithContext(ctx aws.Context, input *kinesis.DescribeStreamInput, opts ...request.Option) (*kinesis.DescribeStreamOutput, error) {
	return nil, notImplementedError
}

func (k *mockKinesis) GetRecordsWithContext(ctx aws.Context, input *kinesis.GetRecordsInput, opts ...request.Option) (*kinesis.GetRecordsOutput, error) {
	return nil, notImplementedError
}

func (k *mockKinesis) PutRecordsWithContext(ctx aws.Context, input *kinesis.PutRecordsInput, opts ...request.Option) (*kinesis.PutRecordsOutput, error) {
	return nil, notImplementedError
}

func newMockKinesis() *mockKinesis {
	return &mockKinesis{}
}

// getRecordsInput is a convenience function for generating a getRecordsInput
func getRecordsInput(limit int64, shardIterator string) *kinesis.GetRecordsInput {
	return &kinesis.GetRecordsInput{
		Limit:         &limit,
		ShardIterator: &shardIterator,
	}
}

// describeStream is a convenience function for generating a *DescribeStreamOutput
func describeStream() *kinesis.DescribeStreamOutput {
	return &kinesis.DescribeStreamOutput{
		StreamDescription: getStreamDescription(),
	}
}

func getStreamDescription() *kinesis.StreamDescription {
	moreShards := false
	streamStatus := "ACTIVE"
	arn := "arn:aws:kinesis:us-west-2/some_stream_name"
	return &kinesis.StreamDescription{
		HasMoreShards: &moreShards,
		Shards:        getStreamShards(),
		StreamARN:     &arn,
		StreamStatus:  &streamStatus,
	}
}

func getStreamShards() []*kinesis.Shard {
	shard1 := "00000000000000000"
	shard2 := "00000000000000001"
	return []*kinesis.Shard{
		{
			AdjacentParentShardId: nil,
			HashKeyRange:          nil,
			ParentShardId:         nil,
			SequenceNumberRange:   nil,
			ShardId:               &shard1,
		},
		{
			AdjacentParentShardId: nil,
			HashKeyRange:          nil,
			ParentShardId:         nil,
			SequenceNumberRange:   nil,
			ShardId:               &shard2,
		},
	}
}

// putRecords is a convenience function for generating a pseudo *PutRecordsOutput
func putRecords() *kinesis.PutRecordsOutput {
	var failed int64
	return &kinesis.PutRecordsOutput{
		FailedRecordCount: &failed,
		Records:           []*kinesis.PutRecordsResultEntry{},
	}
}

func TestKinesis(t *testing.T) {
	k := newMockKinesis()
	ctx := context.Background()
	_, err := k.GetRecordsWithContext(ctx, getRecordsInput(100, "some_shard_iterator"))
	if err == nil {
		t.Fatal("error not generated by mock kinesis")
	}
}
