package kswriter

import (
	"fmt"
)

const (
	LogError LogLevel = iota
	LogWarn
	LogInfo
	LogDebug
	LogTrace
)

type LogLevel int

func (l LogLevel) String() string {
	switch l {
	case LogError:
		return "error"
	case LogWarn:
		return "warn"
	case LogInfo:
		return "info"
	case LogDebug:
		return "debug"
	case LogTrace:
		return "trace"
	default:
		return "undefined"
	}
}

func ParseLogLevel(l string) LogLevel {
	switch l {
	case "error":
		return LogError
	case "warn":
		return LogWarn
	case "info":
		return LogInfo
	case "debug":
		return LogDebug
	case "trace":
		return LogTrace
	default:
		return LogInfo
	}
}

// Logger is the external logger that can be written to by the internal logger
type Logger interface {
	Errorf(format string, v ...interface{})
	Warnf(format string, v ...interface{})
	Infof(format string, v ...interface{})
	Debugf(format string, v ...interface{})
	Tracef(format string, v ...interface{})
}

// logger is the internal used streamlog interface
type logger interface {
	Error(function, message string, v ...interface{})
	Warn(function, message string, v ...interface{})
	Info(function, message string, v ...interface{})
	Debug(function, message string, v ...interface{})
	Trace(function, message string, v ...interface{})
}

type nopLogger struct {
}

func (z *nopLogger) Error(function string, format string, v ...interface{}) {
}

func (z *nopLogger) Warn(function string, format string, v ...interface{}) {
}

func (z *nopLogger) Info(function string, format string, v ...interface{}) {
}

func (z *nopLogger) Debug(function string, format string, v ...interface{}) {
}

func (z *nopLogger) Trace(function string, format string, v ...interface{}) {
}

type streamLogger struct {
	level  LogLevel
	logger Logger
}

func (z *streamLogger) Error(function string, format string, v ...interface{}) {
	if z.level >= LogError {
		z.logger.Errorf(fmt.Sprintf("[%s] %s", function, fmt.Sprintf(format, v...)))
	}
}

func (z *streamLogger) Warn(function string, format string, v ...interface{}) {
	if z.level >= LogWarn {
		z.logger.Warnf(fmt.Sprintf("[%s] %s", function, fmt.Sprintf(format, v...)))
	}
}

func (z *streamLogger) Info(function string, format string, v ...interface{}) {
	if z.level >= LogInfo {
		z.logger.Infof(fmt.Sprintf("[%s] %s", function, fmt.Sprintf(format, v...)))
	}
}

func (z *streamLogger) Debug(function string, format string, v ...interface{}) {
	if z.level >= LogDebug {
		z.logger.Debugf(fmt.Sprintf("[%s] %s", function, fmt.Sprintf(format, v...)))
	}
}

func (z *streamLogger) Trace(function string, format string, v ...interface{}) {
	if z.level >= LogTrace {
		z.logger.Tracef(fmt.Sprintf("[%s] %s", function, fmt.Sprintf(format, v...)))
	}
}
