package kswriter

import (
	"context"
	"errors"
	"fmt"
	"time"
)

const (
	defaultQueueSize            = 2500
	defaultErrorBufferSize      = 1000
	defaultDrainEventQueueTimer = 10 * time.Second
)

// Settings to create a new Streamlog client.
type Settings struct {
	Role                 string // role to assume when creating kinesis client
	Region               string // region
	ServiceTag           string // Name of the service we are logging
	SessionID            string // ID used to group all messages for the same session
	KinesisStreamName    string
	QueueSize            int64
	ErrorBufferSize      int64
	DrainEventQueueTimer time.Duration
	LogLevel             string
	Logger               Logger
	Statter              Statter
}

// New creates a new streamlog client and opens its kinesis connection.
func New(cfg Settings) (*KinesisStreamlogWriter, error) {

	if cfg.Role == "" {
		return nil, errors.New("cannot instantiate KinesisStreamlogWriter, missing Role")
	}

	if cfg.Region == "" {
		return nil, errors.New("cannot instantiate KinesisStreamlogWriter, missing Region")
	}

	if cfg.KinesisStreamName == "" {
		return nil, errors.New("cannot instantiate KinesisStreamlogWriter, missing KinesisStreamName")
	}

	if cfg.ServiceTag == "" {
		return nil, errors.New("cannot instantiate KinesisStreamlogWriter, missing ServiceTag")
	}

	if cfg.QueueSize == 0 {
		cfg.QueueSize = defaultQueueSize
	}

	if cfg.ErrorBufferSize == 0 {
		cfg.ErrorBufferSize = defaultErrorBufferSize
	}

	if cfg.DrainEventQueueTimer == 0 {
		cfg.DrainEventQueueTimer = defaultDrainEventQueueTimer
	}

	var lggr logger
	if cfg.Logger == nil {
		lggr = &nopLogger{}
	} else {
		if cfg.LogLevel == "" {
			cfg.LogLevel = LogWarn.String()
		}
		lggr = &streamLogger{
			level:  ParseLogLevel(cfg.LogLevel),
			logger: cfg.Logger,
		}
	}

	if cfg.Statter == nil {
		cfg.Statter = &NopStatter{}
	}

	ks, err := newKinesis(cfg.Region, cfg.Role)
	if err != nil {
		return nil, fmt.Errorf("cannot instantiate kinesis: %w", err)
	}

	ctx, killSwitch := context.WithCancel(context.Background())

	eh := newErrHandler(ks, cfg, lggr, cfg.Statter)
	go eh.run(ctx)

	ekq := newEventKinesisQ(ks, cfg, eh, lggr, cfg.Statter)
	go ekq.run(ctx)

	writer := &KinesisStreamlogWriter{
		sessionID:  cfg.SessionID,
		serviceTag: cfg.ServiceTag,
		logger:     lggr,
		eventQ:     ekq,
		killSwitch: killSwitch,
	}
	return writer, nil
}
