package wswriter

import (
	"errors"
	"fmt"
	"log"
	"net/http"
	"net/url"
	"time"

	data "code.justin.tv/amzn/streamlogclient/data/v1"
	"github.com/golang/protobuf/proto"
	"github.com/gorilla/websocket"
	"github.com/jpillora/backoff"
)

// Exponential backoff parameters.
const (
	// Min time to wait between reconnect attempts
	minWait = 1 * time.Minute

	// Max time to wait between reconnect attempts
	maxWait = 8 * time.Minute

	// Time to wait between websocket connection recycles
	connRecycleWait = 15 * time.Minute
)

// ErrQueueFull indicates that a message was dropped because the Streamlog client's
// message queue is full.
var ErrQueueFull = errors.New("streamlog queue full. Message not sent")

// WebsocketStreamlogWriter implements the Streamlog interface using `gorilla/websocket` and
// exponential backoff.
type WebsocketStreamlogWriter struct {
	WebsocketEndpoint string             // URL of websocket endpoint; if empty, writes are discarded
	ServiceTag        string             // Name of the service we are logging
	SessionID         string             // ID used to group all messages for the same session
	Proxy             string             // URL of a websocket proxy
	QueueSize         int                // Message buffer size; overflow drops messages, 0 makes Log() synchronous
	queue             chan proto.Message // Events queue'd here. Closed to signal time to quit
}

// SetSessionID updates the default streamlog client's Session ID for all future messages.
// This is required for Delay streams, for example, as their Session IDs are different
// from their original stream.
func (sl *WebsocketStreamlogWriter) SetSessionID(sessionID string) {
	sl.SessionID = sessionID
}

// Log queues a message to send to streamlog. This method returns immediately, regardless
// of whether the message was sent or queued.
// This method will panic if the client's connection has been closed via Close().
func (sl *WebsocketStreamlogWriter) Log(channel, key string, value interface{}) error {
	return sl.LogWithSessionId("", channel, key, value)
}

// LogWithSessionId is like Log, but it uses the provided Session ID instead of the one
// the client was configured with.
// If the `sessionId` argument is the empty string, the client's Session ID is used;
// `LogWithSessionId("", channel, key, value)` is equivalent to `Log(channel, key, value)`
func (sl *WebsocketStreamlogWriter) LogWithSessionId(sessionId, channel, key string, value interface{}) error {
	record := &data.Event_Record{Timestamp: time.Now().Unix()}

	// Fill record data depending on data type
	switch v := value.(type) {
	case int:
		record.I = int64(v)
	case int64:
		record.I = v
	case float32:
		record.F = float64(v)
	case float64:
		record.F = v
	case string:
		record.S = v
	case bool:
		record.B = v
	default:
		return fmt.Errorf("'%T' is not a supported streamlog type", value)
	}

	event := &data.Event{
		Channel:    channel,
		SessionID:  sl.SessionID,
		ServiceTag: sl.ServiceTag,
		Metric:     key,
		Records:    []*data.Event_Record{record},
	}
	if sessionId != "" {
		event.SessionID = sessionId
	}

	// Send the event, but don't block
	select {
	case sl.queue <- event:
		return nil
	default:
		log.Printf("%s: %+v", ErrQueueFull, event)
		return ErrQueueFull
	}
}

// Close the connection to streamlog. This will shut down all helper goroutines.
// Attempts to log more events will panic.
func (sl *WebsocketStreamlogWriter) Close() {
	close(sl.queue)
}

// This function reads from the event queue and writes events to the websocket
// connection. It will attempt to reconnect if the write is unsucessful
func (sl *WebsocketStreamlogWriter) writeLoop() {
	conn := sl.openConnection()
	defer conn.Close()

	// ELB's are recycled every hour - recycle the connection every 15 minutes
	reconnect := time.NewTicker(connRecycleWait)
	defer reconnect.Stop()

	for {
		select {
		case event, ok := <-sl.queue:
			if !ok {
				return // We've been closed
			}

			bytes, err := proto.Marshal(event)
			if err != nil {
				continue
			}

			err = conn.WriteMessage(websocket.BinaryMessage, bytes)
			// Attempt to reconnect and resend until success
			for err != nil {
				log.Printf("Streamlog write failed: %s", err)
				conn = sl.openConnection()
				err = conn.WriteMessage(websocket.BinaryMessage, bytes)
			}

		case <-reconnect.C:
			_ = conn.Close()
			conn = sl.openConnection()
		}
	}
}

// Connect to streamlog websocket endpoint. Will conntinue to attempt to connect
// with exponential backoff if the first attempt isn't successful.
func (sl *WebsocketStreamlogWriter) openConnection() *websocket.Conn {
	header := http.Header{"Origin": {"http://localhost/"}}
	dialer := &websocket.Dialer{
		Proxy: func(req *http.Request) (*url.URL, error) {
			if sl.Proxy == "" {
				return nil, nil
			}
			proxyURL, err := url.Parse(sl.Proxy)
			if err != nil {
				return nil, fmt.Errorf("invalid proxy URL %q. %v", sl.Proxy, err)
			}
			return proxyURL, nil
		},
	}
	conn, _, err := dialer.Dial(sl.WebsocketEndpoint, header)

	// Continue to retry while backing off if there's an error
	back := backoff.Backoff{
		Jitter: true,
		Min:    minWait,
		Max:    maxWait,
	}
	for err != nil {
		waitTime := back.Duration()
		log.Printf("Streamlog failed to connect '%s'. Reconnect in %v. Queue size: %v",
			sl.WebsocketEndpoint, waitTime, len(sl.queue))
		time.Sleep(waitTime)
		conn, _, err = dialer.Dial(sl.WebsocketEndpoint, header)
	}

	log.Printf("Streamlog connected to '%s'. Queue size: %v",
		sl.WebsocketEndpoint, len(sl.queue))

	// Must read to handle ping and close messages from server
	// This is also responsible for closing the connection
	go func(c *websocket.Conn) {
		for {
			if _, _, err := c.NextReader(); err != nil {
				_ = c.Close()
				return
			}
		}
	}(conn)

	return conn
}
