package wswriter

// Streamlog wraps logging events to streamlog. Handles the websocket connection
// for asynchronously sending events to streamlog.

import (
	"errors"

	"github.com/golang/protobuf/proto"
)

// Settings to create a new Streamlog client.
type Settings struct {
	WebsocketEndpoint string // URL of websocket endpoint; if empty, writes are discarded
	ServiceTag        string // Name of the service we are logging
	SessionID         string // ID used to group all messages for the same session
	Proxy             string // URL of a websocket proxy
	QueueSize         int    // Message buffer size; overflow drops messages, 0 makes Log() synchronous
}

// New creates a new streamlog client and opens its connection.
// If the Settings has an empty WebsocketEndpoint, the returned client discards all writes.
func New(cfg Settings) (*WebsocketStreamlogWriter, error) {
	if cfg.WebsocketEndpoint == "" {
		return nil, errors.New("empty websocket endpoint")
	}
	sl := &WebsocketStreamlogWriter{
		WebsocketEndpoint: cfg.WebsocketEndpoint,
		ServiceTag:        cfg.ServiceTag,
		SessionID:         cfg.SessionID,
		Proxy:             cfg.Proxy,
		QueueSize:         cfg.QueueSize,
		queue:             make(chan proto.Message, cfg.QueueSize),
	}
	// Launch worker goroutine that writes to the websocket connection
	go sl.writeLoop()
	return sl, nil
}
