package main

import (
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"time"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/chitin/clog"
	"github.com/zenazn/goji/graceful"
	"golang.org/x/net/context"
)

func init() {
	http.HandleFunc("/", handleHello)
	http.HandleFunc("/sleep", handleSleep)
	http.HandleFunc("/panic", handlePanic)
}

func main() {
	graceful.HandleSignals()

	l, err := net.Listen("tcp", ":8080")
	if err != nil {
		log.Fatalf("bind=%q", err)
	}
	log.Printf("server=%q", "listening")

	ctx, err := chitin.ExperimentalTraceContext(context.Background())
	if err != nil {
		log.Fatalf("trace=%q", err)
	}

	handler := chitin.Handler(http.DefaultServeMux, chitin.SetBaseContext(ctx))
	err = graceful.Serve(l, handler)
	if err != nil {
		log.Fatalf("serve=%q", err)
	}
	log.Printf("server=%q", "draining")
	graceful.Wait()
	log.Printf("server=%q", "done")
}

func handleHello(w http.ResponseWriter, r *http.Request) {
	ctx := chitin.Context(w, r)
	clog.Log(ctx).Printf("url=%q", r.URL)

	_, err := io.Copy(ioutil.Discard, r.Body)
	if err != nil {
		clog.Log(ctx).Printf("body-read-err=%q", err)
		return
	}

	fmt.Fprintf(w, "Hello, %q", r.URL)
}

func handleSleep(w http.ResponseWriter, r *http.Request) {
	time.Sleep(10 * time.Second)
}

func handlePanic(w http.ResponseWriter, r *http.Request) {
	panic("oops")
}
