"""
Errors
======

Exceptions and special cases in Mekansm.
"""


class MekansmException(Exception):
    """Base Exception for Mekansm"""
    resource_type = "Unknown"
    message = "Unknown Error"


class MekansmNotFound(MekansmException):
    """Base Exception for resources that were not found on the context."""
    def __init__(self, id_):
        self.id_ = id_

    @property
    def message(self):
        return "{} not found with ID: {}".format(self.resource_type, self.id_)


class MekansmInvalidId(MekansmException):
    """Base Exception for invalid ID errors."""
    def __init__(self, id_):
        self.id_ = id_

    @property
    def message(self):
        return "Invalid ID for {}: {}".format(self.resource_type, self.id_)


class MekansmInvalidFields(MekansmException):
    """Base Exception for invalid fields errors (missing and unhandled)."""

    def __init__(self, missing_fields, unknown_fields):
        self.missing_fields = missing_fields
        self.unknown_fields = unknown_fields

    @property
    def message(self):
        return "Missing fields: {}. Unknown fields: {}".format(
            list(self.missing_fields), list(self.unknown_fields))


class MekansmInvalidFilters(MekansmException):
    """Base Exception for invalid fields errors (missing and unhandled)."""

    def __init__(self, filters):
        self.filters = filters

    @property
    def message(self):
        return "Invalid filters: {}".format(self.filters)


class DeploymentNotFound(MekansmNotFound):
    """Error when a Deployment couldn't be found in the context"""
    resource_type = "Deployment"


class NodeNotFound(MekansmNotFound):
    """Error when a Node couldn't be found in the context"""
    resource_type = "Node"


class NodeStatusNotFound(MekansmNotFound):
    """Error when a NodeStatus couldn't be found in the context"""
    resource_type = "Node Status"


class InvalidDeploymentId(MekansmInvalidId):
    """Error when a value given is an invalid ID for a Deployment"""
    resource_type = "Deployment"


class InvalidNodeId(MekansmInvalidId):
    """Error when a value given is an invalid ID for a Node"""
    resource_type = "Node"


class InvalidDeploymentFields(MekansmInvalidFields):
    """Error when the given fields are invalid for a Deployment"""
    resource_type = "Deployment"


class InvalidNodeFields(MekansmInvalidFields):
    """Error when the given fields are invalid for a Node"""
    resource_type = "Node"


class InvalidNodeStatusFields(MekansmInvalidFields):
    """Error when the given fields are invalid for a Node"""
    resource_type = "Node Status"


class ProfileNameNotFound(MekansmNotFound):
    """Error when there isn't a configured profile for a deployment"""
    resource_type = "Configured AWS Profile"
