package client

import (
	"bytes"
	"fmt"
	"golang.org/x/net/context"
	"net/http"
	"net/url"
	"strconv"
	"time"
)

// CreateDeploymentPath computes a request path to the create action of deployment.
func CreateDeploymentPath() string {

	return fmt.Sprintf("/deployments")
}

// Creates a new deployment.
func (c *Client) CreateDeployment(ctx context.Context, path string, payload *DeploymentPayload, datetimeFormat *string) (*http.Response, error) {
	req, err := c.NewCreateDeploymentRequest(ctx, path, payload, datetimeFormat)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewCreateDeploymentRequest create the request corresponding to the create action endpoint of the deployment resource.
func (c *Client) NewCreateDeploymentRequest(ctx context.Context, path string, payload *DeploymentPayload, datetimeFormat *string) (*http.Request, error) {
	var body bytes.Buffer
	err := c.Encoder.Encode(payload, &body, "*/*")
	if err != nil {
		return nil, fmt.Errorf("failed to encode body: %s", err)
	}
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("POST", u.String(), &body)
	if err != nil {
		return nil, err
	}
	if c.BasicAuthSigner != nil {
		c.BasicAuthSigner.Sign(req)
	}
	return req, nil
}

// ListDeploymentPath computes a request path to the list action of deployment.
func ListDeploymentPath() string {

	return fmt.Sprintf("/deployments")
}

// Returns a list of deployments.
func (c *Client) ListDeployment(ctx context.Context, path string, datetimeFormat *string, from *time.Time, limit *int, to *time.Time) (*http.Response, error) {
	req, err := c.NewListDeploymentRequest(ctx, path, datetimeFormat, from, limit, to)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewListDeploymentRequest create the request corresponding to the list action endpoint of the deployment resource.
func (c *Client) NewListDeploymentRequest(ctx context.Context, path string, datetimeFormat *string, from *time.Time, limit *int, to *time.Time) (*http.Request, error) {
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	if from != nil {
		tmp14 := from.Format(time.RFC3339)
		values.Set("from", tmp14)
	}
	if limit != nil {
		tmp15 := strconv.Itoa(*limit)
		values.Set("limit", tmp15)
	}
	if to != nil {
		tmp16 := to.Format(time.RFC3339)
		values.Set("to", tmp16)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}
	return req, nil
}

// PreviewDeploymentPath computes a request path to the preview action of deployment.
func PreviewDeploymentPath() string {

	return fmt.Sprintf("/deployments/preview")
}

// Previews the changes done by the creation of a deployment.
func (c *Client) PreviewDeployment(ctx context.Context, path string, payload *DeploymentPayload, datetimeFormat *string) (*http.Response, error) {
	req, err := c.NewPreviewDeploymentRequest(ctx, path, payload, datetimeFormat)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewPreviewDeploymentRequest create the request corresponding to the preview action endpoint of the deployment resource.
func (c *Client) NewPreviewDeploymentRequest(ctx context.Context, path string, payload *DeploymentPayload, datetimeFormat *string) (*http.Request, error) {
	var body bytes.Buffer
	err := c.Encoder.Encode(payload, &body, "*/*")
	if err != nil {
		return nil, fmt.Errorf("failed to encode body: %s", err)
	}
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("POST", u.String(), &body)
	if err != nil {
		return nil, err
	}
	return req, nil
}

// ShowDeploymentPath computes a request path to the show action of deployment.
func ShowDeploymentPath(deploymentID int) string {
	param0 := strconv.Itoa(deploymentID)

	return fmt.Sprintf("/deployments/%s", param0)
}

// Returns detailed information about a deployment.
func (c *Client) ShowDeployment(ctx context.Context, path string, datetimeFormat *string) (*http.Response, error) {
	req, err := c.NewShowDeploymentRequest(ctx, path, datetimeFormat)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewShowDeploymentRequest create the request corresponding to the show action endpoint of the deployment resource.
func (c *Client) NewShowDeploymentRequest(ctx context.Context, path string, datetimeFormat *string) (*http.Request, error) {
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}
	return req, nil
}
