package client

import (
	"fmt"
	"golang.org/x/net/context"
	"net/http"
	"net/url"
	"strconv"
)

// ListEnvironmentPath computes a request path to the list action of environment.
func ListEnvironmentPath(snapshotID int) string {
	param0 := strconv.Itoa(snapshotID)

	return fmt.Sprintf("/snapshots/%s/environments", param0)
}

// Get the contents of all environments in a snapshot.
func (c *Client) ListEnvironment(ctx context.Context, path string, datetimeFormat *string) (*http.Response, error) {
	req, err := c.NewListEnvironmentRequest(ctx, path, datetimeFormat)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewListEnvironmentRequest create the request corresponding to the list action endpoint of the environment resource.
func (c *Client) NewListEnvironmentRequest(ctx context.Context, path string, datetimeFormat *string) (*http.Request, error) {
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}
	return req, nil
}

// ShowEnvironmentPath computes a request path to the show action of environment.
func ShowEnvironmentPath(snapshotID int, environment string) string {
	param0 := strconv.Itoa(snapshotID)
	param1 := environment

	return fmt.Sprintf("/snapshots/%s/environments/%s", param0, param1)
}

// Returns the merged contents of an environment in a snapshot.
func (c *Client) ShowEnvironment(ctx context.Context, path string, datetimeFormat *string) (*http.Response, error) {
	req, err := c.NewShowEnvironmentRequest(ctx, path, datetimeFormat)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewShowEnvironmentRequest create the request corresponding to the show action endpoint of the environment resource.
func (c *Client) NewShowEnvironmentRequest(ctx context.Context, path string, datetimeFormat *string) (*http.Request, error) {
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}
	return req, nil
}
