package client

import (
	"bytes"
	"fmt"
	"golang.org/x/net/context"
	"net/http"
	"net/url"
	"strconv"
	"time"
)

// CreateSnapshotPath computes a request path to the create action of snapshot.
func CreateSnapshotPath() string {

	return fmt.Sprintf("/snapshots")
}

// Creates a new snapshot.
func (c *Client) CreateSnapshot(ctx context.Context, path string, payload *SnapshotPayload, datetimeFormat *string) (*http.Response, error) {
	req, err := c.NewCreateSnapshotRequest(ctx, path, payload, datetimeFormat)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewCreateSnapshotRequest create the request corresponding to the create action endpoint of the snapshot resource.
func (c *Client) NewCreateSnapshotRequest(ctx context.Context, path string, payload *SnapshotPayload, datetimeFormat *string) (*http.Request, error) {
	var body bytes.Buffer
	err := c.Encoder.Encode(payload, &body, "*/*")
	if err != nil {
		return nil, fmt.Errorf("failed to encode body: %s", err)
	}
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("POST", u.String(), &body)
	if err != nil {
		return nil, err
	}
	if c.BasicAuthSigner != nil {
		c.BasicAuthSigner.Sign(req)
	}
	return req, nil
}

// ListSnapshotPath computes a request path to the list action of snapshot.
func ListSnapshotPath() string {

	return fmt.Sprintf("/snapshots")
}

// Get a list of snapshots
func (c *Client) ListSnapshot(ctx context.Context, path string, datetimeFormat *string, from *time.Time, limit *int, to *time.Time) (*http.Response, error) {
	req, err := c.NewListSnapshotRequest(ctx, path, datetimeFormat, from, limit, to)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewListSnapshotRequest create the request corresponding to the list action endpoint of the snapshot resource.
func (c *Client) NewListSnapshotRequest(ctx context.Context, path string, datetimeFormat *string, from *time.Time, limit *int, to *time.Time) (*http.Request, error) {
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	if from != nil {
		tmp17 := from.Format(time.RFC3339)
		values.Set("from", tmp17)
	}
	if limit != nil {
		tmp18 := strconv.Itoa(*limit)
		values.Set("limit", tmp18)
	}
	if to != nil {
		tmp19 := to.Format(time.RFC3339)
		values.Set("to", tmp19)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}
	return req, nil
}

// ShowSnapshotPath computes a request path to the show action of snapshot.
func ShowSnapshotPath(snapshotID int) string {
	param0 := strconv.Itoa(snapshotID)

	return fmt.Sprintf("/snapshots/%s", param0)
}

// Get snapshot by id
func (c *Client) ShowSnapshot(ctx context.Context, path string, datetimeFormat *string) (*http.Response, error) {
	req, err := c.NewShowSnapshotRequest(ctx, path, datetimeFormat)
	if err != nil {
		return nil, err
	}
	return c.Client.Do(ctx, req)
}

// NewShowSnapshotRequest create the request corresponding to the show action endpoint of the snapshot resource.
func (c *Client) NewShowSnapshotRequest(ctx context.Context, path string, datetimeFormat *string) (*http.Request, error) {
	scheme := c.Scheme
	if scheme == "" {
		scheme = "https"
	}
	u := url.URL{Host: c.Host, Scheme: scheme, Path: path}
	values := u.Query()
	if datetimeFormat != nil {
		values.Set("datetime-format", *datetimeFormat)
	}
	u.RawQuery = values.Encode()
	req, err := http.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}
	return req, nil
}
