package design

import (
	. "github.com/goadesign/goa/design" // Use . imports to enable the DSL
	. "github.com/goadesign/goa/design/apidsl"
)

var BasicAuth = BasicAuthSecurity("basic_auth")

var _ = Resource("public", func() {

	Origin("*", func() {
		Methods("GET")
	})

	Files("/swagger.json", "swagger/swagger.json")
	Files("/schema.json", "schema/schema.json")
	Files("/ui/*filepath", "public/swagger-ui/")
})

var _ = Resource("snapshot", func() {
	DefaultMedia(SnapshotMedia)
	BasePath("/snapshots")

	NoSecurity()

	Action("list", func() {
		Routing(GET(""))
		Description("Get a list of snapshots")
		Params(func() {
			Param("limit", Integer, "The number of snapshots is limited by default to 20. This can be changed to a maximum of 100.", func() {
				Default(20)
				Minimum(1)
				Maximum(100)
			})
			Param("from", DateTime, "Return snapshots from this moment.")
			Param("to", DateTime, "Return snapshots before this moment.")
		})
		Response(OK, func() {
			Media(CollectionOf(SnapshotMedia, func() {
				View("listed")
				View("default")
			}))
		})
	})
	Action("show", func() {
		Routing(GET("/:snapshot_id"))
		Description("Get snapshot by id")
		Params(func() {
			Param("snapshot_id", Integer, "Snapshot ID")
		})
		Response(OK)
		Response(NotFound)
	})
	Action("create", func() {
		Routing(POST(""))
		Description("Creates a new snapshot.")
		Payload(SnapshotPayload)
		Response(Created, "/snapshots/.+")
		Response(Unauthorized)
		Security(BasicAuth)
	})
})

var _ = Resource("environment", func() {
	DefaultMedia(EnvironmentMedia)
	BasePath("/snapshots/:snapshot_id/environments")
	Params(func() {
		Param("snapshot_id", Integer, "Snapshot ID")
	})
	NoSecurity()
	Action("list", func() {
		Routing(GET(""))
		Description("Get the contents of all environments in a snapshot.")
		Response(OK, func() {
			Media(CollectionOf(EnvironmentMedia, func() {
				View("default")
			}))
		})
		Response(NotFound)
	})
	Action("show", func() {
		Routing(GET("/:environment"))
		Description("Returns the merged contents of an environment in a snapshot.")
		Params(func() {
			Param("environment", String)
		})
		Response(OK)
		Response(NotFound)
	})
})

var _ = Resource("deployment", func() {
	DefaultMedia(DeploymentMedia)
	BasePath("/deployments")
	NoSecurity()

	Action("list", func() {
		Routing(GET(""))
		Description("Returns a list of deployments.")
		Params(func() {
			Param("limit", Integer, "The number of deployments is limited by default to 20. This can be changed to a maximum of 100.", func() {
				Default(20)
				Minimum(1)
				Maximum(100)
			})
			Param("from", DateTime, "Return deployments from this moment.")
			Param("to", DateTime, "Return deployments before this moment.")
		})
		Response(OK, CollectionOf(DeploymentMedia))
	})
	Action("show", func() {
		Routing(GET("/:deployment_id"))
		Description("Returns detailed information about a deployment.")
		Params(func() {
			Param("deployment_id", Integer, "Deployment ID")
		})
		Response(OK)
		Response(NotFound)
	})
	Action("create", func() {
		Routing(POST(""))
		Description("Creates a new deployment.")
		Payload(DeploymentPayload)
		Response(Created, "/deployments/.+")
		Response(BadRequest, ErrorMedia)
		Response(Unauthorized)
		Security(BasicAuth)
	})
	Action("preview", func() {
		Routing(POST("/preview"))
		Description("Previews the changes done by the creation of a deployment.")
		Payload(DeploymentPayload)
		Response(OK, SnapshotStateMapMedia)
		Response(BadRequest, ErrorMedia)
	})
})
