package cli

import (
	"code.justin.tv/dta/necronomicon-user-api/client"
	"encoding/json"
	"fmt"
	"github.com/goadesign/goa"
	goaclient "github.com/goadesign/goa/client"
	uuid "github.com/goadesign/goa/uuid"
	"github.com/spf13/cobra"
	"golang.org/x/net/context"
	"log"
	"net/url"
	"os"
	"path"
	"strconv"
	"strings"
	"time"
)

type (
	// CreateDeploymentCommand is the command line data structure for the create action of deployment
	CreateDeploymentCommand struct {
		Payload     string
		ContentType string
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		PrettyPrint    bool
	}

	// ListDeploymentCommand is the command line data structure for the list action of deployment
	ListDeploymentCommand struct {
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		// Return deployments from this moment.
		From string
		// The number of deployments is limited by default to 20. This can be changed to a maximum of 100.
		Limit int
		// Return deployments before this moment.
		To          string
		PrettyPrint bool
	}

	// PreviewDeploymentCommand is the command line data structure for the preview action of deployment
	PreviewDeploymentCommand struct {
		Payload     string
		ContentType string
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		PrettyPrint    bool
	}

	// ShowDeploymentCommand is the command line data structure for the show action of deployment
	ShowDeploymentCommand struct {
		// Deployment ID
		DeploymentID int
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		PrettyPrint    bool
	}

	// ListEnvironmentCommand is the command line data structure for the list action of environment
	ListEnvironmentCommand struct {
		// Snapshot ID
		SnapshotID int
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		PrettyPrint    bool
	}

	// ShowEnvironmentCommand is the command line data structure for the show action of environment
	ShowEnvironmentCommand struct {
		Environment string
		// Snapshot ID
		SnapshotID int
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		PrettyPrint    bool
	}

	// CreateSnapshotCommand is the command line data structure for the create action of snapshot
	CreateSnapshotCommand struct {
		Payload     string
		ContentType string
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		PrettyPrint    bool
	}

	// ListSnapshotCommand is the command line data structure for the list action of snapshot
	ListSnapshotCommand struct {
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		// Return snapshots from this moment.
		From string
		// The number of snapshots is limited by default to 20. This can be changed to a maximum of 100.
		Limit int
		// Return snapshots before this moment.
		To          string
		PrettyPrint bool
	}

	// ShowSnapshotCommand is the command line data structure for the show action of snapshot
	ShowSnapshotCommand struct {
		// Snapshot ID
		SnapshotID int
		// Timestamps are returned in the format requested.
		DatetimeFormat string
		PrettyPrint    bool
	}

	// DownloadCommand is the command line data structure for the download command.
	DownloadCommand struct {
		// OutFile is the path to the download output file.
		OutFile string
	}
)

// RegisterCommands registers the resource action CLI commands.
func RegisterCommands(app *cobra.Command, c *client.Client) {
	var command, sub *cobra.Command
	command = &cobra.Command{
		Use:   "create",
		Short: `create action`,
	}
	tmp1 := new(CreateDeploymentCommand)
	sub = &cobra.Command{
		Use:   `deployment ["/deployments"]`,
		Short: ``,
		Long: `

Payload example:

{
   "message": "Odit exercitationem atque magnam dolore.",
   "snapshot_id": 361726276988422645
}`,
		RunE: func(cmd *cobra.Command, args []string) error { return tmp1.Run(c, args) },
	}
	tmp1.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp1.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	tmp2 := new(CreateSnapshotCommand)
	sub = &cobra.Command{
		Use:   `snapshot ["/snapshots"]`,
		Short: ``,
		Long: `

Payload example:

{
   "environments": [
      {
         "name": "Odit quasi aut corrupti non modi.",
         "operation": "deleted",
         "parent": "Quia cupiditate voluptas laboriosam qui.",
         "values": {
            "Rem qui.": {
               "Quis et.": "2001-12-20T14:49:27-08:00"
            }
         }
      },
      {
         "name": "Odit quasi aut corrupti non modi.",
         "operation": "deleted",
         "parent": "Quia cupiditate voluptas laboriosam qui.",
         "values": {
            "Rem qui.": {
               "Quis et.": "2001-12-20T14:49:27-08:00"
            }
         }
      }
   ],
   "message": "A ut doloribus."
}`,
		RunE: func(cmd *cobra.Command, args []string) error { return tmp2.Run(c, args) },
	}
	tmp2.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp2.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	app.AddCommand(command)
	command = &cobra.Command{
		Use:   "list",
		Short: `list action`,
	}
	tmp3 := new(ListDeploymentCommand)
	sub = &cobra.Command{
		Use:   `deployment ["/deployments"]`,
		Short: ``,
		RunE:  func(cmd *cobra.Command, args []string) error { return tmp3.Run(c, args) },
	}
	tmp3.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp3.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	tmp4 := new(ListEnvironmentCommand)
	sub = &cobra.Command{
		Use:   `environment ["/snapshots/SNAPSHOT_ID/environments"]`,
		Short: ``,
		RunE:  func(cmd *cobra.Command, args []string) error { return tmp4.Run(c, args) },
	}
	tmp4.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp4.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	tmp5 := new(ListSnapshotCommand)
	sub = &cobra.Command{
		Use:   `snapshot ["/snapshots"]`,
		Short: ``,
		RunE:  func(cmd *cobra.Command, args []string) error { return tmp5.Run(c, args) },
	}
	tmp5.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp5.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	app.AddCommand(command)
	command = &cobra.Command{
		Use:   "preview",
		Short: `Previews the changes done by the creation of a deployment.`,
	}
	tmp6 := new(PreviewDeploymentCommand)
	sub = &cobra.Command{
		Use:   `deployment ["/deployments/preview"]`,
		Short: ``,
		Long: `

Payload example:

{
   "message": "Odit exercitationem atque magnam dolore.",
   "snapshot_id": 361726276988422645
}`,
		RunE: func(cmd *cobra.Command, args []string) error { return tmp6.Run(c, args) },
	}
	tmp6.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp6.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	app.AddCommand(command)
	command = &cobra.Command{
		Use:   "show",
		Short: `show action`,
	}
	tmp7 := new(ShowDeploymentCommand)
	sub = &cobra.Command{
		Use:   `deployment ["/deployments/DEPLOYMENT_ID"]`,
		Short: ``,
		RunE:  func(cmd *cobra.Command, args []string) error { return tmp7.Run(c, args) },
	}
	tmp7.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp7.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	tmp8 := new(ShowEnvironmentCommand)
	sub = &cobra.Command{
		Use:   `environment ["/snapshots/SNAPSHOT_ID/environments/ENVIRONMENT"]`,
		Short: ``,
		RunE:  func(cmd *cobra.Command, args []string) error { return tmp8.Run(c, args) },
	}
	tmp8.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp8.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	tmp9 := new(ShowSnapshotCommand)
	sub = &cobra.Command{
		Use:   `snapshot ["/snapshots/SNAPSHOT_ID"]`,
		Short: ``,
		RunE:  func(cmd *cobra.Command, args []string) error { return tmp9.Run(c, args) },
	}
	tmp9.RegisterFlags(sub, c)
	sub.PersistentFlags().BoolVar(&tmp9.PrettyPrint, "pp", false, "Pretty print response body")
	command.AddCommand(sub)
	app.AddCommand(command)

	dl := new(DownloadCommand)
	dlc := &cobra.Command{
		Use:   "download [PATH]",
		Short: "Download file with given path",
		RunE: func(cmd *cobra.Command, args []string) error {
			return dl.Run(c, args)
		},
	}
	dlc.Flags().StringVar(&dl.OutFile, "out", "", "Output file")
	app.AddCommand(dlc)
}

func intFlagVal(name string, parsed int) *int {
	if hasFlag(name) {
		return &parsed
	}
	return nil
}

func float64FlagVal(name string, parsed float64) *float64 {
	if hasFlag(name) {
		return &parsed
	}
	return nil
}

func boolFlagVal(name string, parsed bool) *bool {
	if hasFlag(name) {
		return &parsed
	}
	return nil
}

func stringFlagVal(name string, parsed string) *string {
	if hasFlag(name) {
		return &parsed
	}
	return nil
}

func hasFlag(name string) bool {
	for _, arg := range os.Args[1:] {
		if strings.HasPrefix(arg, "--"+name) {
			return true
		}
	}
	return false
}

func jsonVal(val string) (*interface{}, error) {
	var t interface{}
	err := json.Unmarshal([]byte(val), &t)
	if err != nil {
		return nil, err
	}
	return &t, nil
}

func jsonArray(ins []string) ([]interface{}, error) {
	if ins == nil {
		return nil, nil
	}
	var vals []interface{}
	for _, id := range ins {
		val, err := jsonVal(id)
		if err != nil {
			return nil, err
		}
		vals = append(vals, val)
	}
	return vals, nil
}

func timeVal(val string) (*time.Time, error) {
	t, err := time.Parse(time.RFC3339, val)
	if err != nil {
		return nil, err
	}
	return &t, nil
}

func timeArray(ins []string) ([]time.Time, error) {
	if ins == nil {
		return nil, nil
	}
	var vals []time.Time
	for _, id := range ins {
		val, err := timeVal(id)
		if err != nil {
			return nil, err
		}
		vals = append(vals, *val)
	}
	return vals, nil
}

func uuidVal(val string) (*uuid.UUID, error) {
	t, err := uuid.FromString(val)
	if err != nil {
		return nil, err
	}
	return &t, nil
}

func uuidArray(ins []string) ([]uuid.UUID, error) {
	if ins == nil {
		return nil, nil
	}
	var vals []uuid.UUID
	for _, id := range ins {
		val, err := uuidVal(id)
		if err != nil {
			return nil, err
		}
		vals = append(vals, *val)
	}
	return vals, nil
}

func float64Val(val string) (*float64, error) {
	t, err := strconv.ParseFloat(val, 64)
	if err != nil {
		return nil, err
	}
	return &t, nil
}

func float64Array(ins []string) ([]float64, error) {
	if ins == nil {
		return nil, nil
	}
	var vals []float64
	for _, id := range ins {
		val, err := float64Val(id)
		if err != nil {
			return nil, err
		}
		vals = append(vals, *val)
	}
	return vals, nil
}

func boolVal(val string) (*bool, error) {
	t, err := strconv.ParseBool(val)
	if err != nil {
		return nil, err
	}
	return &t, nil
}

func boolArray(ins []string) ([]bool, error) {
	if ins == nil {
		return nil, nil
	}
	var vals []bool
	for _, id := range ins {
		val, err := boolVal(id)
		if err != nil {
			return nil, err
		}
		vals = append(vals, *val)
	}
	return vals, nil
}

// Run downloads files with given paths.
func (cmd *DownloadCommand) Run(c *client.Client, args []string) error {
	var (
		fnf func(context.Context, string) (int64, error)
		fnd func(context.Context, string, string) (int64, error)

		rpath   = args[0]
		outfile = cmd.OutFile
		logger  = goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
		ctx     = goa.WithLogger(context.Background(), logger)
		err     error
	)

	if rpath[0] != '/' {
		rpath = "/" + rpath
	}
	if rpath == "/schema.json" {
		fnf = c.DownloadSchemaJSON
		if outfile == "" {
			outfile = "schema.json"
		}
		goto found
	}
	if rpath == "/swagger.json" {
		fnf = c.DownloadSwaggerJSON
		if outfile == "" {
			outfile = "swagger.json"
		}
		goto found
	}
	if strings.HasPrefix(rpath, "/ui/") {
		fnd = c.DownloadUI
		rpath = rpath[4:]
		if outfile == "" {
			_, outfile = path.Split(rpath)
		}
		goto found
	}
	return fmt.Errorf("don't know how to download %s", rpath)
found:
	ctx = goa.WithLogContext(ctx, "file", outfile)
	if fnf != nil {
		_, err = fnf(ctx, outfile)
	} else {
		_, err = fnd(ctx, rpath, outfile)
	}
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	return nil
}

// Run makes the HTTP request corresponding to the CreateDeploymentCommand command.
func (cmd *CreateDeploymentCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = "/deployments"
	}
	var payload client.DeploymentPayload
	if cmd.Payload != "" {
		err := json.Unmarshal([]byte(cmd.Payload), &payload)
		if err != nil {
			return fmt.Errorf("failed to deserialize payload: %s", err)
		}
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	resp, err := c.CreateDeployment(ctx, path, &payload, stringFlagVal("datetime-format", cmd.DatetimeFormat))
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *CreateDeploymentCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	cc.Flags().StringVar(&cmd.Payload, "payload", "", "Request body encoded in JSON")
	cc.Flags().StringVar(&cmd.ContentType, "content", "", "Request content type override, e.g. 'application/x-www-form-urlencoded'")
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
}

// Run makes the HTTP request corresponding to the ListDeploymentCommand command.
func (cmd *ListDeploymentCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = "/deployments"
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	var tmp10 *time.Time
	if cmd.From != "" {
		var err error
		tmp10, err = timeVal(cmd.From)
		if err != nil {
			goa.LogError(ctx, "failed to parse flag into *time.Time value", "flag", "--from", "err", err)
			return err
		}
	}
	var tmp11 *time.Time
	if cmd.To != "" {
		var err error
		tmp11, err = timeVal(cmd.To)
		if err != nil {
			goa.LogError(ctx, "failed to parse flag into *time.Time value", "flag", "--to", "err", err)
			return err
		}
	}
	resp, err := c.ListDeployment(ctx, path, stringFlagVal("datetime-format", cmd.DatetimeFormat), tmp10, intFlagVal("limit", cmd.Limit), tmp11)
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *ListDeploymentCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
	var from string
	cc.Flags().StringVar(&cmd.From, "from", from, `Return deployments from this moment.`)
	cc.Flags().IntVar(&cmd.Limit, "limit", 20, `The number of deployments is limited by default to 20. This can be changed to a maximum of 100.`)
	var to string
	cc.Flags().StringVar(&cmd.To, "to", to, `Return deployments before this moment.`)
}

// Run makes the HTTP request corresponding to the PreviewDeploymentCommand command.
func (cmd *PreviewDeploymentCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = "/deployments/preview"
	}
	var payload client.DeploymentPayload
	if cmd.Payload != "" {
		err := json.Unmarshal([]byte(cmd.Payload), &payload)
		if err != nil {
			return fmt.Errorf("failed to deserialize payload: %s", err)
		}
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	resp, err := c.PreviewDeployment(ctx, path, &payload, stringFlagVal("datetime-format", cmd.DatetimeFormat))
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *PreviewDeploymentCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	cc.Flags().StringVar(&cmd.Payload, "payload", "", "Request body encoded in JSON")
	cc.Flags().StringVar(&cmd.ContentType, "content", "", "Request content type override, e.g. 'application/x-www-form-urlencoded'")
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
}

// Run makes the HTTP request corresponding to the ShowDeploymentCommand command.
func (cmd *ShowDeploymentCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = fmt.Sprintf("/deployments/%v", cmd.DeploymentID)
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	resp, err := c.ShowDeployment(ctx, path, stringFlagVal("datetime-format", cmd.DatetimeFormat))
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *ShowDeploymentCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	var deploymentID int
	cc.Flags().IntVar(&cmd.DeploymentID, "deployment_id", deploymentID, `Deployment ID`)
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
}

// Run makes the HTTP request corresponding to the ListEnvironmentCommand command.
func (cmd *ListEnvironmentCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = fmt.Sprintf("/snapshots/%v/environments", cmd.SnapshotID)
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	resp, err := c.ListEnvironment(ctx, path, stringFlagVal("datetime-format", cmd.DatetimeFormat))
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *ListEnvironmentCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	var snapshotID int
	cc.Flags().IntVar(&cmd.SnapshotID, "snapshot_id", snapshotID, `Snapshot ID`)
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
}

// Run makes the HTTP request corresponding to the ShowEnvironmentCommand command.
func (cmd *ShowEnvironmentCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = fmt.Sprintf("/snapshots/%v/environments/%v", cmd.SnapshotID, url.QueryEscape(cmd.Environment))
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	resp, err := c.ShowEnvironment(ctx, path, stringFlagVal("datetime-format", cmd.DatetimeFormat))
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *ShowEnvironmentCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	var environment string
	cc.Flags().StringVar(&cmd.Environment, "environment", environment, ``)
	var snapshotID int
	cc.Flags().IntVar(&cmd.SnapshotID, "snapshot_id", snapshotID, `Snapshot ID`)
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
}

// Run makes the HTTP request corresponding to the CreateSnapshotCommand command.
func (cmd *CreateSnapshotCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = "/snapshots"
	}
	var payload client.SnapshotPayload
	if cmd.Payload != "" {
		err := json.Unmarshal([]byte(cmd.Payload), &payload)
		if err != nil {
			return fmt.Errorf("failed to deserialize payload: %s", err)
		}
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	resp, err := c.CreateSnapshot(ctx, path, &payload, stringFlagVal("datetime-format", cmd.DatetimeFormat))
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *CreateSnapshotCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	cc.Flags().StringVar(&cmd.Payload, "payload", "", "Request body encoded in JSON")
	cc.Flags().StringVar(&cmd.ContentType, "content", "", "Request content type override, e.g. 'application/x-www-form-urlencoded'")
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
}

// Run makes the HTTP request corresponding to the ListSnapshotCommand command.
func (cmd *ListSnapshotCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = "/snapshots"
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	var tmp12 *time.Time
	if cmd.From != "" {
		var err error
		tmp12, err = timeVal(cmd.From)
		if err != nil {
			goa.LogError(ctx, "failed to parse flag into *time.Time value", "flag", "--from", "err", err)
			return err
		}
	}
	var tmp13 *time.Time
	if cmd.To != "" {
		var err error
		tmp13, err = timeVal(cmd.To)
		if err != nil {
			goa.LogError(ctx, "failed to parse flag into *time.Time value", "flag", "--to", "err", err)
			return err
		}
	}
	resp, err := c.ListSnapshot(ctx, path, stringFlagVal("datetime-format", cmd.DatetimeFormat), tmp12, intFlagVal("limit", cmd.Limit), tmp13)
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *ListSnapshotCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
	var from string
	cc.Flags().StringVar(&cmd.From, "from", from, `Return snapshots from this moment.`)
	cc.Flags().IntVar(&cmd.Limit, "limit", 20, `The number of snapshots is limited by default to 20. This can be changed to a maximum of 100.`)
	var to string
	cc.Flags().StringVar(&cmd.To, "to", to, `Return snapshots before this moment.`)
}

// Run makes the HTTP request corresponding to the ShowSnapshotCommand command.
func (cmd *ShowSnapshotCommand) Run(c *client.Client, args []string) error {
	var path string
	if len(args) > 0 {
		path = args[0]
	} else {
		path = fmt.Sprintf("/snapshots/%v", cmd.SnapshotID)
	}
	logger := goa.NewLogger(log.New(os.Stderr, "", log.LstdFlags))
	ctx := goa.WithLogger(context.Background(), logger)
	resp, err := c.ShowSnapshot(ctx, path, stringFlagVal("datetime-format", cmd.DatetimeFormat))
	if err != nil {
		goa.LogError(ctx, "failed", "err", err)
		return err
	}

	goaclient.HandleResponse(c.Client, resp, cmd.PrettyPrint)
	return nil
}

// RegisterFlags registers the command flags with the command line.
func (cmd *ShowSnapshotCommand) RegisterFlags(cc *cobra.Command, c *client.Client) {
	var snapshotID int
	cc.Flags().IntVar(&cmd.SnapshotID, "snapshot_id", snapshotID, `Snapshot ID`)
	var datetimeFormat string
	cc.Flags().StringVar(&cmd.DatetimeFormat, "datetime-format", datetimeFormat, `Timestamps are returned in the format requested.`)
}
