"""Unit tests for code.justin.tv.dta.rockpaperscissors.event_store module."""

from __future__ import absolute_import

import base64
import logging
import unittest

from boto3.dynamodb import types
import mock

from . import event_bus
from . import event_store
from . import large_event_store


logging.basicConfig(level=logging.INFO)


class EventStoreTestCase(unittest.TestCase):

    def testBasic(self):
        mock_large_event_store = mock.create_autospec(
            large_event_store.LargeEventStore, instance=True)
        store = event_store.EventStore('table_name', mock_large_event_store)
        event = event_bus.Event(
          uuid='abcdef',
          timestamp=0.0,
          event_type='type',
          body='body',
          attributes={'key': 'value'})

        with mock.patch.object(store, 'dynamodb', autospec=True):
            mock_table = store.dynamodb.Table.return_value
            mock_batch = mock_table.batch_writer.return_value
            mock_batch.__enter__.return_value = mock_batch

            store.PutEvents([event])

            store.dynamodb.Table.assert_called_once_with('table_name')
            mock_batch.put_item.assert_called_once_with(
              Item={
                'uuid': types.Binary('abcdef'),
                'timestamp': types.Decimal(0.0),
                'type': 'type',
                'body': types.Binary(event.SerializeToString()),
                'attributes': {
                  'key': 'value',
                },
              })

            mock_batch.reset_mock()
            event.body = 'a' * event_store.MAX_ITEM_BYTES
            store.PutEvents([event])
            mock_large_event_store.PutEvent.assert_called_once_with(event)
            mock_batch.put_item.assert_not_called()


if __name__ == '__main__':
    unittest.main()
