# Adding New Routes

Routes are composed of the following - when adding a new route you'll need to create these where applicable.

## Route Definition
Every route should have an `index.js` that defines your route.  You'll also need to modify the `routes/index.js` to import your route as a child route.  We're defining our routes as [PlainRoute](https://github.com/ReactTraining/react-router/blob/master/docs/API.md#plainroute) objects like so:

```js
import { injectReducer } from '../../store/reducers'

export default (store) => ({
  // This is the '/' url-path that you can access your route at.  If you have params you also put them here
  path: 'teams(/:serviceId)',

  // Define the actual component your route points to and attach it to a reducer if needed.
  getComponent (nextState, next) {
    require.ensure([], (require) => {
      const component = require('./containers/TeamInfo').default  // Actual component.
      const reducer = require('./reducers/team_info').default     // Reducer to connect to our state.

      // This adds your reducer to the state.  It'll be accessible at the given key.  In this case, `state.team_info`.
      injectReducer(store, {
        key: 'team_info',
        reducer: reducer
      })
      next(null, component)
    })
  }
})
```

### Adding sub-routes to routes
It's possible that routes have other routes within.  If this is the case for a route, then simply add the sub-route as a child of the route via defining childRoutes in the Route.

## Route Component
Ultimately your route needs to point to a component - so make sure you define one.  Typically the components for a route should go in `components` directory within your route's directory.  Contents of your component can be whatever you want.

### Styling components
If you need custom styling for your component then you can define a `.scss` file in your directory and import it.  You can then apply it to elements via the `className` property.  Example:

```js
// Contents of `Home/components/Home.scss`
@import 'base';
.homeWelcomePanel {
  border-color: $twitch-purple-7;
}

// Contents of `Home/components/Home.js`
import React from 'react'
import { Panel } from 'react-bootstrap'
import RpsImage from '../assets/rps.png'
import './Home.scss'
export default class HomeView extends React.Component {
  render () {
    const header = <div className='text-center teamHeader'>Welcome to RPS</div>
    return (
      <Panel className='homeWelcomePanel' header={header} >
        <Image className='center-block' src={RpsImage} responsive />
      </Panel>
    )
  }
}
```

### Adding assets
Note in the above example we've imported the path to an image and passed it to an Image Component.  If you have any assets your component needs, you can do a similar thing to get the asset's relative path to components.

## Route Container
If your route needs to interface with the application's state, then you should wrap your Route Component in a Container and pass the Container to the route.  The Container will map the application state to the component's properties.

```js
// Contents of `HelloWorld/containers/HelloWorld.js`
import { connect } from 'react-redux'
import HelloWorld from '../components'

const mapStateToProps = (state, ownProps) => {
  return {
    metrics: state.api_data.metrics,  //HelloWorld can access this at `this.props.metrics`
  }
}

export default connect(mapStateToProps)(HelloWorld)

// Contents of `HelloWorld/components/HelloWorld.js`
render () {
  let metrics = []
  Object.keys(this.props.metrics).forEach((metricId) => {
    metrics.push(<li>metricId</li>)
  })
  return (  
    <div>Hello World!</div>
    <ul>
      {metrics}
    </ul>
  )
}
```
