import React, { Component } from 'react'
import { Glyphicon } from 'react-bootstrap'
import { Link } from 'react-router'

class NestedTableRow extends Component {

  constructor (props) {
    super(props)

    // Using our level property let's compute our padding level and also determine our parent levels.
    // Do this so we can store it in our state and not have to recompute it everytime.
    let parentLevelsValue = []
    let curParentStr = ''
    for (let c of props.id) {
      if (c === '.') {
        parentLevelsValue.push(curParentStr)
      }
      curParentStr += c
    }

    // Determine if we have any children.
    const hasChildRows = (props.childRows.length > 0)

    // Add the stuff we computed above to our state, so we don't have to compute it again on future renders.
    this.state = {
      parentLevels: parentLevelsValue,
      hasChildRows: hasChildRows,
      expanded: false
    }

    // This binding is necessary to make `this` work in the callback.
    this.onExpand = this.onExpand.bind(this)
  }

  // By default we're expanded and not hidden.
  defaultProps = {
    onExpand: null
  }

  /**
   * Called when a expand button is clicked, calls our onExpand() prop and passes it our level.
   */
  onExpand () {
    let state = Object.assign({}, this.state)
    state.expanded = !state.expanded
    this.setState(state)
    this.props.onExpand(this.props.id)
  }

  /**
   * Generates our expand button if one is needed.  Will also toggle the state of the expand button based on whether
   * we're expanded or not.
   */
  getExpandButton () {
    if (this.props.onExpand === null) {
      return null
    }
    const disabled = this.state.disabled
    let glyphImg = 'chevron-right'
    if (disabled || this.state.expanded) {
      glyphImg = 'chevron-down'
    }
    let glyphClassName = 'expand-glyph'
    let glyph = <Glyphicon glyph={glyphImg} style={{ top: '2px' }} className={glyphClassName} />
    if (!disabled) {
      glyph = <a onClick={this.onExpand}>{glyph}</a>
    }
    return glyph
  }

  /**
   * Iterates through our list of parent levels and checks to see if any of them are hidden.  If
   * they are, then we are too.
   */
  isHidden () {
    const expandedRows = this.props.expandedRows
    let hiddenRows = Object.assign([], this.state.parentLevels)
    for (var i = 0; i < this.state.parentLevels.length; i++) {
      const rowLevel = this.state.parentLevels[i]
      if (expandedRows[rowLevel]) {
        hiddenRows.splice(hiddenRows.indexOf(rowLevel), 1)
      }
    }
    if (hiddenRows.length > 0) {
      return true
    }
    return false
  }

  /**
   * Renders our component.
   */
  render () {
    let content = this.props.content
    if (this.props.url) {
      content = <Link to={this.props.url}>{content}</Link>
    }
    content = <td className={`content`}>{content}</td>

    let expandButton = this.getExpandButton()
    let className = 'expand-button'
    expandButton = <td className={className}>{expandButton}</td>

    let rowContent = <tr>{expandButton}{content}</tr>
    if (this.props.level === 0) {
      rowContent = <tr>{content}{expandButton}</tr>
    }
    rowContent = (
      <td className={`level-${this.props.level} first-col`}>
        <table className={`first-col-table level-${this.props.level}`}>
          <tbody>
            {rowContent}
          </tbody>
        </table>
      </td>
    )

    if (this.isHidden()) {
      return null
    }
    return (
      <tr>
        {rowContent}
        {this.props.children}
      </tr>
    )
  }
}

NestedTableRow.propTypes = {
  id: React.PropTypes.string.isRequired,
  level: React.PropTypes.number.isRequired,
  content: React.PropTypes.string,
  onExpand: React.PropTypes.func,
  expandedRows: React.PropTypes.object.isRequired,
  children: React.PropTypes.any,
  url: React.PropTypes.string,
  childRows: React.PropTypes.array
}

export default NestedTableRow
