package metrics

import (
	"context"
	"math"

	structpb "github.com/golang/protobuf/ptypes/struct"
	"github.com/montanaflynn/stats"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"

	"code.justin.tv/dta/rockpaperscissors/internal/codecov"
	pb "code.justin.tv/dta/rockpaperscissors/proto"
)

func init() {
	info := &pb.MetricInfo{
		MetricId:        "test_coverage_percent",
		MetricName:      "% Test Coverage",
		Description:     "Median percent of test coverage in Jenkins jobs.",
		ValidForProject: true,
		KeyMetric:       true,
	}
	Registry().Register(info, NewTestCoveragePercentCalculator)
}

// TestCoveragePercentCalculator to calculate median tests passing in Jenkins jobs.
type TestCoveragePercentCalculator struct {
	ProjectMetadataServer pb.ProjectMetadataServiceServer
	EventServer           pb.EventServiceServer
}

// NewTestCoveragePercentCalculator factory for TestCoveragePercentCalculator structs.
func NewTestCoveragePercentCalculator(projectMetadataServer pb.ProjectMetadataServiceServer, eventServer pb.EventServiceServer) (Calculator, error) {
	return &TestCoveragePercentCalculator{
		ProjectMetadataServer: projectMetadataServer,
		EventServer:           eventServer,
	}, nil
}

func (p *TestCoveragePercentCalculator) getDataPointForEvent(event *pb.Event) (float64, error) {
	format := "coverage.xml" // The default before codecov_coverage.json was added.
	for _, attr := range event.GetAttributes() {
		if attr.GetKey() == "coverage_file" {
			format = attr.GetValue()
			break
		}
	}

	switch format {
	case "codecov_coverage.json":
		coverageReport, err := codecov.Unmarshal(event.GetBody())
		if err != nil {
			return 0.0, errf(codes.Internal,
				"Error parsing test results JUnit XML: %v", err)
		}
		return coverageReport.Commit.Totals.CoverageRatio / 100.0, nil
	default:
		return math.NaN(), nil
	}
}

func (p *TestCoveragePercentCalculator) getDataPointsForJob(ctx context.Context, jenkinsJob string, timerange *pb.TimeRange) ([]float64, error) {
	var dataPoints []float64

	queryResp, err := p.EventServer.QueryEvents(ctx, &pb.QueryEventsRequest{
		Timerange: timerange,
		Type:      "JenkinsTestCoverage",
		Filters: []*pb.QueryEventsRequest_AttributeFilter{
			&pb.QueryEventsRequest_AttributeFilter{
				Key:   "jenkins_job_name",
				Value: jenkinsJob,
			},
		},
	})
	if err != nil {
		if grpc.Code(err) == codes.NotFound {
			return nil, nil
		}
		return nil, err
	}
	for _, event := range queryResp.GetEvents() {
		dataPoint, err := p.getDataPointForEvent(event)
		if err != nil {
			// TODO: it might be better to just log the error and skip it
			return nil, err
		}
		if !(math.IsNaN(dataPoint) || math.IsInf(dataPoint, 0)) {
			dataPoints = append(dataPoints, dataPoint)
		}
	}

	return dataPoints, nil
}

func (p *TestCoveragePercentCalculator) calculateEntry(ctx context.Context, req *pb.GetMetricRequest, entry *pb.GetMetricResponse_TimeSeriesEntry, jenkinsJobs []string) error {
	var dataPoints []float64

	for _, jenkinsJob := range jenkinsJobs {
		jobDataPoints, err := p.getDataPointsForJob(
			ctx, jenkinsJob, entry.GetTimerange())
		if err != nil {
			return err
		}
		if jobDataPoints != nil {
			dataPoints = append(dataPoints, jobDataPoints...)
		}
	}

	if len(dataPoints) == 0 {
		return nil
	}

	median, err := stats.Median(dataPoints)
	if err != nil {
		return errf(codes.Internal, "Error calculating median value: %v", err)
	}

	entry.Value = &structpb.Value{
		Kind: &structpb.Value_NumberValue{
			NumberValue: median,
		},
	}

	return nil
}

// Calculate the time series and fill out the response.
func (p *TestCoveragePercentCalculator) Calculate(ctx context.Context, req *pb.GetMetricRequest, resp *pb.GetMetricResponse) error {
	timeSeries, err := makeTimeSeries(req.Timerange, req.BucketSize, req.IanaTimeZone)
	if err != nil {
		return errf(codes.InvalidArgument,
			"Found inappropriate time range: %v", err)
	}
	resp.TimeSeries = timeSeries
	resp.TimeSeriesUnits = "%"

	jenkinsJobs, err := getJenkinsJobsForProject(ctx, p.ProjectMetadataServer, req.ProjectId)
	if err != nil {
		return err
	}
	if len(jenkinsJobs) == 0 {
		return nil
	}

	sem := make(chan error, len(timeSeries))
	for _, bucket := range timeSeries {
		go func(ctx context.Context, req *pb.GetMetricRequest, entry *pb.GetMetricResponse_TimeSeriesEntry, jenkinsJobs []string) {
			sem <- p.calculateEntry(ctx, req, entry, jenkinsJobs)
		}(ctx, req, bucket, jenkinsJobs)
	}
	for i := 0; i < len(timeSeries); i++ {
		err := <-sem
		if err != nil {
			// TODO: cancel the context to stop any still-processing work?
			return err
		}
	}

	return nil
}
