package metrics

import (
	"context"
	"strings"

	structpb "github.com/golang/protobuf/ptypes/struct"
	gh "github.com/google/go-github/github"
	"github.com/montanaflynn/stats"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"

	pb "code.justin.tv/dta/rockpaperscissors/proto"
)

func init() {
	info := &pb.MetricInfo{
		MetricId:        "num_prod_deploys",
		MetricName:      "# of Prod Deploys",
		Description:     "Number of successful deployments through Skadi to production environments",
		ValidForProject: true,
		KeyMetric:       true,
	}
	Registry().Register(info, NewNumProdDeploysCalculator)
}

// NumProdDeploysCalculator to calculate how many Jenkins builds a project made.
type NumProdDeploysCalculator struct {
	ProjectMetadataServer pb.ProjectMetadataServiceServer
	EventServer           pb.EventServiceServer
}

// NewNumProdDeploysCalculator factory for NumProdDeploysCalculator structs.
func NewNumProdDeploysCalculator(projectMetadataServer pb.ProjectMetadataServiceServer, eventServer pb.EventServiceServer) (Calculator, error) {
	return &NumProdDeploysCalculator{
		ProjectMetadataServer: projectMetadataServer,
		EventServer:           eventServer,
	}, nil
}

func (p *NumProdDeploysCalculator) getDataPointForEvent(event *pb.Event) (float64, error) {
	githubEvent, err := getGitHubEvent(event, "deployment_status")
	if err != nil {
		return 0.0, err
	}

	switch githubEvent := githubEvent.(type) {
	case *gh.DeploymentStatusEvent:
		if githubEvent.Deployment == nil {
			return 0.0, nil
		}
		if strings.HasPrefix(githubEvent.Deployment.GetEnvironment(), "prod") {
			return 1.0, nil
		}
	}
	return 0.0, nil
}

func (p *NumProdDeploysCalculator) getDataPointsForRepo(ctx context.Context, githubRepo string, timerange *pb.TimeRange) ([]float64, error) {
	var dataPoints []float64

	queryResp, err := p.EventServer.QueryEvents(ctx, &pb.QueryEventsRequest{
		Timerange: timerange,
		Type:      "GitHub-deployment_status",
		Filters: []*pb.QueryEventsRequest_AttributeFilter{
			&pb.QueryEventsRequest_AttributeFilter{
				Key:   "github_repository",
				Value: githubRepo,
			},
			&pb.QueryEventsRequest_AttributeFilter{
				Key:   "deployment_state",
				Value: "success",
			},
		},
	})
	if err != nil {
		if grpc.Code(err) == codes.NotFound {
			return nil, nil
		}
		return nil, err
	}
	for _, event := range queryResp.GetEvents() {
		dataPoint, err := p.getDataPointForEvent(event)
		if err != nil {
			// TODO: it might be better to just log the error and skip it
			return nil, err
		}
		dataPoints = append(dataPoints, dataPoint)
	}

	return dataPoints, nil
}

func (p *NumProdDeploysCalculator) calculateEntry(ctx context.Context, req *pb.GetMetricRequest, entry *pb.GetMetricResponse_TimeSeriesEntry, githubRepos []string) error {
	var dataPoints []float64

	for _, githubRepo := range githubRepos {
		repoDataPoints, err := p.getDataPointsForRepo(
			ctx, githubRepo, entry.GetTimerange())
		if err != nil {
			return err
		}
		if repoDataPoints != nil {
			dataPoints = append(dataPoints, repoDataPoints...)
		}
	}

	if len(dataPoints) == 0 {
		return nil
	}

	sum, err := stats.Sum(dataPoints)
	if err != nil {
		return errf(codes.Internal, "Error calculating sum: %v", err)
	}

	entry.Value = &structpb.Value{
		Kind: &structpb.Value_NumberValue{
			NumberValue: sum,
		},
	}

	return nil
}

// Calculate the time series and fill out the response.
func (p *NumProdDeploysCalculator) Calculate(ctx context.Context, req *pb.GetMetricRequest, resp *pb.GetMetricResponse) error {
	timeSeries, err := makeTimeSeries(req.Timerange, req.BucketSize, req.IanaTimeZone)
	if err != nil {
		return errf(codes.InvalidArgument,
			"Found inappropriate time range: %v", err)
	}
	resp.TimeSeries = timeSeries
	resp.TimeSeriesUnits = "Deployments"

	githubRepos, err := getGitHubReposForProject(ctx, p.ProjectMetadataServer, req.ProjectId)
	if err != nil {
		return err
	}
	if len(githubRepos) == 0 {
		return nil
	}

	sem := make(chan error, len(timeSeries))
	for _, bucket := range timeSeries {
		go func(ctx context.Context, req *pb.GetMetricRequest, entry *pb.GetMetricResponse_TimeSeriesEntry, githubRepos []string) {
			sem <- p.calculateEntry(ctx, req, entry, githubRepos)
		}(ctx, req, bucket, githubRepos)
	}
	for i := 0; i < len(timeSeries); i++ {
		err := <-sem
		if err != nil {
			// TODO: cancel the context to stop any still-processing work?
			return err
		}
	}

	return nil
}
