package metrics

import (
	"context"

	"github.com/golang/protobuf/proto"
	structpb "github.com/golang/protobuf/ptypes/struct"
	"github.com/montanaflynn/stats"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"

	pb "code.justin.tv/dta/rockpaperscissors/proto"
	eventspb "code.justin.tv/dta/rockpaperscissors/proto/events"
)

func init() {
	info := &pb.MetricInfo{
		MetricId:        "sloc_changed",
		MetricName:      "SLOC Changed",
		Description:     "Lines of code changed in GitHub",
		ValidForProject: true,
		KeyMetric:       true,
	}
	Registry().Register(info, NewSLOCChangedCalculator)
}

// SLOCChangedCalculator to calculate code churn from GitHub push events.
type SLOCChangedCalculator struct {
	ProjectMetadataServer pb.ProjectMetadataServiceServer
	EventServer           pb.EventServiceServer
}

// NewSLOCChangedCalculator factory for SLOCChangedCalculator structs.
func NewSLOCChangedCalculator(projectMetadataServer pb.ProjectMetadataServiceServer, eventServer pb.EventServiceServer) (Calculator, error) {
	return &SLOCChangedCalculator{
		ProjectMetadataServer: projectMetadataServer,
		EventServer:           eventServer,
	}, nil
}

func (p *SLOCChangedCalculator) getDataPointForEvent(event *pb.Event) (float64, error) {
	statsEvent := &eventspb.GitHubCommitStatsEvent{}
	err := proto.Unmarshal(event.Body, statsEvent)
	if err != nil {
		return 0.0, errf(codes.Internal,
			"Error unmarshalling GitHub stats event: %v", err)
	}
	return float64(statsEvent.GetTotal()), nil
}

func (p *SLOCChangedCalculator) getDataPointsForRepo(ctx context.Context, githubRepo string, timerange *pb.TimeRange) ([]float64, error) {
	var dataPoints []float64

	queryResp, err := p.EventServer.QueryEvents(ctx, &pb.QueryEventsRequest{
		Timerange: timerange,
		Type:      "GitHub-Stats",
		Filters: []*pb.QueryEventsRequest_AttributeFilter{
			&pb.QueryEventsRequest_AttributeFilter{
				Key:   "github_repository",
				Value: githubRepo,
			},
		},
	})
	if err != nil {
		if grpc.Code(err) == codes.NotFound {
			return nil, nil
		}
		return nil, err
	}
	for _, event := range queryResp.GetEvents() {
		dataPoint, err := p.getDataPointForEvent(event)
		if err != nil {
			// TODO: it might be better to just log the error and skip it
			return nil, err
		}
		dataPoints = append(dataPoints, dataPoint)
	}

	return dataPoints, nil
}

func (p *SLOCChangedCalculator) calculateEntry(ctx context.Context, req *pb.GetMetricRequest, entry *pb.GetMetricResponse_TimeSeriesEntry, githubRepos []string) error {
	var dataPoints []float64

	for _, githubRepo := range githubRepos {
		repoDataPoints, err := p.getDataPointsForRepo(
			ctx, githubRepo, entry.GetTimerange())
		if err != nil {
			return err
		}
		if repoDataPoints != nil {
			dataPoints = append(dataPoints, repoDataPoints...)
		}
	}

	if len(dataPoints) == 0 {
		return nil
	}

	sum, err := stats.Sum(dataPoints)
	if err != nil {
		return errf(codes.Internal, "Error calculating sum: %v", err)
	}

	entry.Value = &structpb.Value{
		Kind: &structpb.Value_NumberValue{
			NumberValue: sum,
		},
	}

	return nil
}

// Calculate the code churn time series and fill out the response.
func (p *SLOCChangedCalculator) Calculate(ctx context.Context, req *pb.GetMetricRequest, resp *pb.GetMetricResponse) error {
	timeSeries, err := makeTimeSeries(req.Timerange, req.BucketSize, req.IanaTimeZone)
	if err != nil {
		return errf(codes.InvalidArgument,
			"Found inappropriate time range: %v", err)
	}
	resp.TimeSeries = timeSeries
	resp.TimeSeriesUnits = "SLOC"

	githubRepos, err := getGitHubReposForProject(ctx, p.ProjectMetadataServer, req.ProjectId)
	if err != nil {
		return err
	}
	if len(githubRepos) == 0 {
		return nil
	}

	sem := make(chan error, len(timeSeries))
	for _, bucket := range timeSeries {
		go func(ctx context.Context, req *pb.GetMetricRequest, entry *pb.GetMetricResponse_TimeSeriesEntry, githubRepos []string) {
			sem <- p.calculateEntry(ctx, req, entry, githubRepos)
		}(ctx, req, bucket, githubRepos)
	}
	for i := 0; i < len(timeSeries); i++ {
		err := <-sem
		if err != nil {
			// TODO: cancel the context to stop any still-processing work?
			return err
		}
	}

	return nil
}
