# Event stream.
# The writers will be event sources such as GitHub, Jenkins, and Skadi.
# Reader will be a lambda function that copies events to rockpaperscissors backend.
resource "aws_kinesis_stream" "event_stream" {
  name        = "${var.name}-${var.environment}-events"
  shard_count = 1

  tags {
    Name        = "${var.name}-${var.environment}"
    Owner       = "${var.owner}"
    Service     = "${var.project}"
    Environment = "${var.environment}"
  }
}

# Datastore for event objects.
resource "aws_dynamodb_table" "events_table" {
  # The first two components of the name of the table needs to match that of
  # the lmabda function name.
  name = "${var.name}-${var.environment}-events"

  read_capacity  = "${var.events_table_read_capacity}"
  write_capacity = "${var.events_table_write_capacity}"
  hash_key       = "uuid"
  range_key      = "timestamp"

  attribute {
    name = "uuid"
    type = "B"
  }

  attribute {
    name = "timestamp"
    type = "N"
  }

  attribute {
    name = "type"
    type = "S"
  }

  global_secondary_index {
    name               = "EventTypeIndex"
    hash_key           = "type"
    range_key          = "timestamp"
    read_capacity      = "${var.eventtype_gsi_read_capacity}"
    write_capacity     = "${var.eventtype_gsi_write_capacity}"
    projection_type    = "INCLUDE"
    non_key_attributes = ["uuid", "body", "attributes"]
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Name        = "${var.name}-${var.environment}"
    Owner       = "${var.owner}"
    Service     = "${var.project}"
    Environment = "${var.environment}"
  }
}

# Datastore for project metadata.
resource "aws_dynamodb_table" "projects_table" {
  name           = "${var.name}-${var.environment}-projects"
  read_capacity  = "${var.projects_table_read_capacity}"
  write_capacity = "${var.projects_table_write_capacity}"
  hash_key       = "project_id"

  attribute {
    name = "project_id"
    type = "S"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Name        = "${var.name}-${var.environment}"
    Owner       = "${var.owner}"
    Service     = "${var.project}"
    Environment = "${var.environment}"
  }
}

# Lambda function that subscribes to the kinesis stream and writes events to the
# events table.
module "tf_lambda_app" {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/dta/tf_lambda_app.git?ref=1.4.1"
  owner          = "${var.owner}"
  repository     = "${var.project}"
  environment    = "${var.environment}"
  principal      = "${var.account_id}"
  lambda_handler = "event_store.lambda_handler"

  # The lambda function name needs to match that of first two components of
  # the dynamo table name.
  lambda_function_name = "${var.name}-${var.environment}"

  lambda_description = "Receives events from a kinesis stream and writes them to a dynamodb table."
  lambda_runtime     = "python2.7"

  lambda_timeout = "30"

  lambda_vpc_subnet_ids         = "${var.subnet_ids}"
  lambda_vpc_security_group_ids = "${var.security_groups}"
}

# Have the lambda receive events from the kinesis stream.
resource "aws_lambda_event_source_mapping" "event_source_mapping" {
  batch_size        = 20
  event_source_arn  = "${aws_kinesis_stream.event_stream.arn}"
  enabled           = true
  function_name     = "${module.tf_lambda_app.arn}"
  starting_position = "TRIM_HORIZON"
}

resource "aws_sqs_queue" "ingest_queue" {
  name = "${var.name}-${var.environment}-ingest"

  lifecycle {
    prevent_destroy = true
  }
}

resource "aws_s3_bucket" "s3_bucket" {
  bucket = "${var.name}-${var.environment}"
  acl    = "private"

  tags {
    Name        = "${var.name}-${var.environment}"
    Owner       = "${var.owner}"
    Service     = "${var.project}"
    Environment = "${var.environment}"
  }
}

resource "aws_elasticache_subnet_group" "memcached_subnet_group" {
  name       = "${var.name}-${var.environment}"
  subnet_ids = ["${split(",", var.subnet_ids)}"]
}

resource "aws_elasticache_cluster" "memcached" {
  cluster_id           = "${coalesce(var.memcached_cluster_name, null_resource.vars.triggers.default_memcache_cluster_name)}"
  engine               = "memcached"
  node_type            = "cache.t2.micro"
  port                 = 11211
  parameter_group_name = "default.memcached1.4"

  num_cache_nodes    = 3
  az_mode            = "cross-az"
  availability_zones = ["us-west-2a", "us-west-2b", "us-west-2c"]

  subnet_group_name  = "${aws_elasticache_subnet_group.memcached_subnet_group.name}"
  security_group_ids = ["${split(",", var.security_groups)}"]

  tags {
    Name        = "${var.name}-${var.environment}"
    Owner       = "${var.owner}"
    Service     = "${var.project}"
    Environment = "${var.environment}"
  }
}
