package structs

import "time"

type AppType interface {
	NewRows(string, string) ([]*Row, error)
}

type DashboardRequest struct {
	Dashboard *Dashboard `json:"dashboard"`
	Overwrite bool       `json:"overwrite"`
	Meta      *Meta      `json:"meta,omitempty"`
}

type Meta struct {
	Type    string    `json:"type"`
	CanSave bool      `json:"canSave"`
	CanEdit bool      `json:"canEdit"`
	CanStar bool      `json:"canStar"`
	Slug    string    `json:"slug"`
	Expires time.Time `json:"expires"`
	Created time.Time `json:"created"`
}

type Dashboard struct {
	ID            *int       `json:"id"`
	Title         string     `json:"title"`
	Tags          []string   `json:"tags"`
	Timezone      string     `json:"timezone"`
	Refresh       string     `json:"refresh"`
	Rows          []*Row     `json:"rows"`
	SchemaVersion int        `json:"schemaVersion"`
	Templating    Templates  `json:"templating"`
	Time          TimeWindow `json:"time"`
	Version       int        `json:"version"`

	Annotations *Annotations `json:"annotations,omitempty"`
}

type Annotations struct {
	List []Annotation `json:"list"`
}

type Annotation struct {
	Datasource string `json:"datasource"`
	Enable     bool   `json:"enable"`
	IconColor  string `json:"iconColor"`
	IconSize   int    `json:"iconSize"`
	LineColor  string `json:"lineColor"`
	Name       string `json:"name"`
	ShowLine   bool   `json:"showLine"`
	Target     string `json:"target"`
}

type Row struct {
	Collapse bool     `json:"collapse"`
	Editable bool     `json:"editable"`
	Height   string   `json:"height"`
	Panels   []*Panel `json:"panels"`
	Title    string   `json:"title"`
}

type Panel struct {
	AliasColors     struct{}   `json:"aliasColors"`
	Bars            bool       `json:"bars"`
	Datasource      *string    `json:"datasource"`
	Editable        bool       `json:"editable"`
	Error           bool       `json:"error"`
	Fill            int        `json:"fill"`
	Grid            *Grid      `json:"grid,omitempty"`
	ID              int        `json:"id"`
	LeftYAxisLabel  string     `json:"leftYAxisLabel"`
	Legend          Legend     `json:"legend"`
	Lines           bool       `json:"lines"`
	LineWidth       int        `json:"linewidth"`
	Links           []struct{} `json:"links,omitempty"`
	NullPointMode   string     `json:"nullPointMode"`
	Percentage      bool       `json:"percentage"`
	PointRadius     int        `json:"pointradius"`
	Points          bool       `json:"points"`
	Renderer        string     `json:"renderer"`
	RightYAxisLabel string     `json:"rightYAxisLabel"`
	SeriesOverrides []Override `json:"seriesOverrides"`
	Span            int        `json:"span"`
	Stack           bool       `json:"stack"`
	SteppedLine     bool       `json:"steppedLine"`
	Targets         []Target   `json:"targets"`
	TimeFrom        *struct{}  `json:"timeFrom,omitempty"`
	TimeShift       *struct{}  `json:"timeShift,omitempty"`
	Title           string     `json:"title"`
	Tooltip         Tooltip    `json:"tooltip"`
	Type            string     `json:"type"`
	XAxis           bool       `json:"x-axis"`
	YAxis           bool       `json:"y-axis"`
	YFormats        []string   `json:"y_formats"`
}

type Grid struct {
	LeftLogBase     int      `json:"leftLogBase"`
	LeftMax         *int     `json:"leftMax"`
	LeftMin         *int     `json:"leftMin"`
	RightLogBase    int      `json:"rightLogBase"`
	RightMax        *int     `json:"rightMax"`
	RightMin        *int     `json:"rightMin"`
	Threshold1      struct{} `json:"threshold1"`
	Threshold1Color string   `json:"threshold1Color"`
	Threshold2      struct{} `json:"threshold2"`
	Threshold2Color string   `json:"threshold2Color"`
}

type Legend struct {
	Avg     bool `json:"avg"`
	Current bool `json:"current"`
	Max     bool `json:"max"`
	Min     bool `json:"min"`
	Show    bool `json:"show"`
	Total   bool `json:"total"`
	Values  bool `json:"values"`
}

type Override struct {
	Alias string `json:"alias"`
	YAxis int    `json:"yaxis"`
}

type Target struct {
	Dimensions *Dimensions `json:"dimensions"`
	Hide       bool        `json:"hide"`
	MetricName string      `json:"metricName"`
	Namespace  string      `json:"namespace"`
	RefID      string      `json:"refId"`
	Region     string      `json:"region"`
	Statistics []string    `json:"statistics"`
	Target     string      `json:"target"`
}

type Dimensions struct {
	LoadBalancerName string `json:"LoadBalancerName"`
}

type Tooltip struct {
	Shared    bool   `json:"shared"`
	ValueType string `json:"value_type"`
}

type TimeWindow struct {
	From string `json:"from"`
	To   string `json:"to"`
}

type Templates struct {
	List []TemplateVar `json:"list"`
}

type TemplateVar struct {
	Query      string `json:"query"`
	Datasource string `json:"datasource"`
	Name       string `json:"name"`
	Type       string `json:"type"`
	Refresh    bool   `json:"refresh"`
}

func NewPanel(id int, span int, title, y1Label, y2Label string, showLegend bool, targets []Target, datasource string) *Panel {
	p := &Panel{
		Datasource: &datasource,
		Editable:   true,
		Fill:       1,
		Grid: &Grid{
			LeftLogBase:  1,
			RightLogBase: 1,
		},
		ID:             id,
		LeftYAxisLabel: y1Label,
		Legend: Legend{
			Show: showLegend,
		},
		Lines:           true,
		LineWidth:       2,
		NullPointMode:   "null as zero",
		PointRadius:     5,
		Renderer:        "flot",
		RightYAxisLabel: y2Label,
		SeriesOverrides: []Override{},
		Span:            span,
		Targets:         targets,
		Title:           title,
		Tooltip: Tooltip{
			Shared:    true,
			ValueType: "cumulative",
		},
		Type:     "graph",
		XAxis:    true,
		YAxis:    true,
		YFormats: []string{"short", "short"},
	}

	return p
}
