package main

import (
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"syscall"
	"time"

	"code.justin.tv/availability/buckytools/fill"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
)

/*
Global S3 variables of type S3Uploader/S3Downloader (implemented by real S3 vars as well as mock S3 vars)
*/
var uploader S3Uploader
var downloader S3Downloader

//interface implemented by s3manager.Uploader and mockS3Uploader
type S3Uploader interface {
	Upload(*s3manager.UploadInput, ...func(*s3manager.Uploader)) (*s3manager.UploadOutput, error)
}

//interface implemented by s3manager.Downloader and mockS3Downloader
type S3Downloader interface {
	Download(w io.WriterAt, input *s3.GetObjectInput, options ...func(*s3manager.Downloader)) (n int64, err error)
}

/*
Print usage info
*/
func usage() {
	fmt.Printf("%s -bucket=<bucket> -prefix=<prefix> -days=<days>\n", os.Args[0])
	fmt.Printf("\tWalks <prefix> directory and uploads stale .wsp files to S3 <bucket>.\n")
	fmt.Printf("\tFiles with m-time older than (time.now - <days> days) qualify as stale.\n")
	fmt.Printf("\tFiles already existing in <bucket> are backfilled and reuploaded.\n")
	fmt.Printf("\tUploaded files are removed and empty directories are recursively deleted.\n")
}

/*
Sets up S3 vars
Receives uploader/downloader of type S3Uploader/S3Downloader (interface defined above)- either real s3 vars, or mock s3 vars
*/
func setUpS3(uploaderIn S3Uploader, downloaderIn S3Downloader) {
	uploader = uploaderIn
	downloader = downloaderIn
}

/*
Intiliazes S3 variables and calls setUpS3() and then archive()
*/
func main() {
	bucket := flag.String("bucket", "", "S3 bucket name")
	prefix := flag.String("prefix", "", "Directory prefix")
	days := flag.Int("days", -1, "Files with m-time older than (time.now - [days] days) will be removed")
	flag.Parse()

	if *bucket == "" || *prefix == "" || *days == -1 {
		usage()
		os.Exit(1)
	}

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2")},
	)
	if err != nil {
		log.Println(err)
		return
	}
	uploader := s3manager.NewUploader(sess)
	downloader := s3manager.NewDownloader(sess)

	setUpS3(uploader, downloader)
	archive(*prefix, *bucket, *days)
}

/*
Checks if a time is "stale"- older than 30 days is stale
*/
func isStale(check time.Time, days int) bool {
	return check.Before(time.Now().AddDate(0, 0, -1*days))
}

/*
Helper that backfills localPath whisper file with tempPath whisper file data
Returns error if fail, else nil
*/
func backfillLocalWsp(tempPath string, localPath string) error {
	err := fill.All(tempPath, localPath)
	if err != nil {
		return err
	}
	fmt.Println("Merged", tempPath, "and", localPath)
	return nil
}

/*
Uploads an input file to the specified s3 bucket
Returns err if fail, else nil
*/
func uploadToS3(file *os.File, path string, bucket string) error {
	_, err := uploader.Upload(&s3manager.UploadInput{
		Body:   file,
		Bucket: aws.String(bucket),
		Key:    aws.String(path),
	})
	if err != nil {
		return err
	}
	return nil
}

/*
Removes an input file from disk (file's path is also passed in for conv)
Attempts to recursively remove parent folders while parents are empty
Returns err if fail, else nil
*/
func removeFile(file *os.File, path string) error {
	var fileCount int = 0
	var dirCount int = 0
	dir := filepath.Dir(path)
	//removes leaf file
	err := os.Remove(path)
	if err != nil {
		return err
	}
	fileCount += 1
	for {
		//recursively removes parent dir if it can
		err = os.Remove(dir)
		if err != nil {
			e, ok := err.(*os.PathError)
			if ok && e.Err == syscall.ENOTEMPTY {
				//directory not empty
				break
			} else {
				return err
			}
		}
		dirCount += 1
		dir = filepath.Dir(dir)
	}
	fmt.Println("Removed", fileCount, "file(s) and", dirCount, "directorie(s).")
	return nil
}

/*
Checks whether a path exists in s3 bucket
-If file is in bucket, we need to backfill new data:
downloads s3 contents into temp file and returns (tempFile, nil),
tempFile is later merged with local file and the local file is reuploaded
-If file is not in bucket, returns (nil, nil)
-If some other error s caught, returns (nil, err)
*/
func fileInS3(path string, bucket string) (*os.File, error) {
	tempFile, err := ioutil.TempFile(os.TempDir(), "temp")
	if err != nil {
		return nil, err
	}
	_, err = downloader.Download(tempFile,
		&s3.GetObjectInput{
			Bucket: aws.String(bucket),
			Key:    aws.String(path),
		})
	if err == nil {
		return tempFile, nil
	} else {
		if aerr, ok := err.(awserr.Error); ok {
			switch aerr.Code() {
			case s3.ErrCodeNoSuchKey:
				defer os.Remove(tempFile.Name())
				return nil, nil
			default:
			}
		}
		defer os.Remove(tempFile.Name())
		return nil, err
	}
}

/*
Main function: Takes path of outer most parent and bucket
Walks entire parent directory, looking for stale, leaf .wsp files
When found, checks if file is already in s3, if so:
	backfill s3 -> local
	upload local
	remove local
If not already in s3:
	upload local
	remove local
Internal walk will log errors and continue unless fleInS3 fails,
in which case we break and return as this means a bigger AWS issue
*/
func archive(dirPath string, bucket string, days int) {
	filepath.Walk(dirPath, func(path string, info os.FileInfo, err error) error {
		if !info.IsDir() {
			extension := filepath.Ext(path)
			if (extension == ".wsp" || extension == ".txt") && isStale(info.ModTime(), days) {
				file, err := os.Open(path)
				if err != nil {
					log.Println(err)
					return nil
				}
				defer file.Close()

				tempFile, err := fileInS3(path, bucket) //refer to fileInS3() signature for details on how tempFile, err are handled
				if tempFile != nil {
					defer os.Remove(tempFile.Name())
					err = backfillLocalWsp(tempFile.Name(), path)
					if err != nil {
						log.Println(err)
						return nil
					}
				}
				if err == nil {
					err = uploadToS3(file, path, bucket)
					if err != nil {
						log.Println(err)
						return nil
					}
					err = removeFile(file, path)
					if err != nil {
						log.Println(err)
						return nil
					}
				} else {
					//leave the directory walk, bc this error is AWS related and needs to be fixed
					log.Println(err)
					return err
				}
			}
		}
		return nil
	})
}
