package main

import (
	"io"
	"io/ioutil"
	"os"
	"path/filepath"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

var bucket string

type mockS3Uploader struct {
	mock.Mock
}
type mockS3Downloader struct {
	mock.Mock
}

func (m *mockS3Uploader) Upload(input *s3manager.UploadInput, options ...func(*s3manager.Uploader)) (*s3manager.UploadOutput, error) {
	args := m.Called(input)
	return args.Get(0).(*s3manager.UploadOutput), args.Error(1)
}

func (m *mockS3Downloader) Download(w io.WriterAt, input *s3.GetObjectInput, options ...func(*s3manager.Downloader)) (n int64, err error) {
	args := m.Called(w, input)
	return args.Get(0).(int64), args.Error(1)
}

func TestMain(m *testing.M) {
	bucket = "nwbhatia-test"
	os.Exit(m.Run())
}

func TestFileInS3(t *testing.T) {
	mu := &mockS3Uploader{}
	md := &mockS3Downloader{}
	setUpS3(mu, md)

	file1, err := ioutil.TempFile(os.TempDir(), "temp1")
	file2, err := ioutil.TempFile(os.TempDir(), "temp2")
	file3, err := ioutil.TempFile(os.TempDir(), "temp3")

	path1, err := filepath.Abs(file1.Name())
	path2, err := filepath.Abs(file2.Name())
	path3, err := filepath.Abs(file3.Name())

	defer os.Remove(file1.Name())
	defer os.Remove(file2.Name())
	defer os.Remove(file3.Name())

	md.On("Download", mock.AnythingOfType("*os.File"), &s3.GetObjectInput{Bucket: aws.String(bucket), Key: aws.String(path1)}).Return(int64(10), nil)
	md.On("Download", mock.AnythingOfType("*os.File"), &s3.GetObjectInput{Bucket: aws.String(bucket), Key: aws.String(path2)}).Return(int64(10), awserr.New(s3.ErrCodeNoSuchKey, "test", nil))
	md.On("Download", mock.AnythingOfType("*os.File"), &s3.GetObjectInput{Bucket: aws.String(bucket), Key: aws.String(path3)}).Return(int64(10), awserr.New(s3.ErrCodeNoSuchBucket, "test", nil))

	tFile, err := fileInS3(path1, bucket)
	defer os.Remove(tFile.Name())
	assert.NotNil(t, tFile)
	assert.Nil(t, err)

	tFile, err = fileInS3(path2, bucket)
	assert.Nil(t, tFile)
	assert.Nil(t, err)

	tFile, err = fileInS3(path3, bucket)
	assert.Nil(t, tFile)
	assert.NotNil(t, err)

	md.AssertExpectations(t)
}

func TestRemove(t *testing.T) {
	mu := &mockS3Uploader{}
	md := &mockS3Downloader{}
	setUpS3(mu, md)

	_, err := os.Stat("test")
	if err == nil {
		os.RemoveAll("test")
	}
	createDirectory("test", 3, 2, 1)
	staleFiles := make(map[string]int)
	filepath.Walk("test", func(path string, info os.FileInfo, err error) error {
		if !info.IsDir() && filepath.Ext(path) == ".txt" {
			//we found a .txt file
			if isStale(info.ModTime(), 30) {
				staleFiles[path] = 1
				// fmt.Println("Added %s", path)
			}
		}
		return nil
	})

	md.On("Download", mock.AnythingOfType("*os.File"), mock.AnythingOfType("*s3.GetObjectInput")).Return(int64(10), awserr.New(s3.ErrCodeNoSuchKey, "test", nil))
	mu.On("Upload", mock.AnythingOfType("*s3manager.UploadInput")).Return(&s3manager.UploadOutput{}, nil)

	archive("test", bucket, 30)

	filepath.Walk("test", func(path string, info os.FileInfo, err error) error {
		//if leaf file, make sure it is not one of the stale files (these should be removed)
		if !info.IsDir() && filepath.Ext(path) == ".txt" {
			if _, ok := staleFiles[path]; ok {
				t.Error("Stale file was not removed??")
			}
		}
		//if folder, make sure folder is not empty
		if info.IsDir() {
			f, err := os.Open(path)
			if err != nil {
				t.Error("Couldn't open folder path...")
			}
			defer f.Close()

			_, err = f.Readdir(1) // Or f.Readdir(1)
			if err == io.EOF {
				t.Error("Empty folder was not removed...")
			}
		}
		return nil
	})
	removeDirectory("test")

	mu.AssertExpectations(t)
	md.AssertExpectations(t)
}

func TestStale(t *testing.T) {
	assert := assert.New(t)
	assert.Equal(isStale(time.Now(), 30), false)
	assert.Equal(isStale(time.Now().AddDate(10, 10, 10), 30), false)
	assert.Equal(isStale(time.Now().AddDate(0, -2, -1), 30), true)
}

func TestUpload(t *testing.T) {
	mu := &mockS3Uploader{}
	md := &mockS3Downloader{}
	setUpS3(mu, md)

	file, err := ioutil.TempFile(os.TempDir(), "temp")
	if err != nil {
		t.Error(err)
	}

	path, err := filepath.Abs(file.Name())
	if err != nil {
		t.Error(err)
	}

	defer os.Remove(file.Name())

	correctParams := &s3manager.UploadInput{
		Body:   file,
		Bucket: aws.String(bucket),
		Key:    aws.String(path),
	}

	mu.On("Upload", correctParams).Return(&s3manager.UploadOutput{}, nil)

	err = uploadToS3(file, path, bucket)
	if err != nil {
		t.Error(err)
	}

	mu.AssertExpectations(t)
}

// var sess *session.Session
// var svc *s3.S3
// var uploader *s3manager.Uploader
// var downloader *s3manager.Downloader
// var bucket string

// func TestMain(m *testing.M) {
// 	sess, _ = session.NewSession(&aws.Config{
// 		Region: aws.String("us-west-2")},
// 	)
// 	svc = s3.New(sess)
// 	downloader = s3manager.NewDownloader(sess)
// 	uploader = s3manager.NewUploader(sess)
// 	bucket = "nwbhatia-test"

// 	os.Exit(m.Run())
// }

// func TestMerge(t *testing.T) {
// 	//cant really test this without .wsp files?
// }

// func TestS3(t *testing.T) {
// 	file, err := ioutil.TempFile(os.TempDir(), "temp")
// 	if err != nil {
// 		panic(err)
// 	}

// 	path, err := filepath.Abs(file.Name())
// 	if err != nil {
// 		t.Error(err)
// 	}

// 	fmt.Println("The file path is: ", path)

// 	defer os.Remove(file.Name())

// 	err = uploadToS3(file, path, bucket, uploader)
// 	if err != nil {
// 		t.Error(err)
// 	}

// 	// Get the list of items
// 	resp, err := svc.ListObjects(&s3.ListObjectsInput{Bucket: aws.String(bucket)})
// 	if err != nil {
// 		t.Error("Unable to list items in bucket %q, %v", bucket, err)
// 	}

// 	for _, item := range resp.Contents {
// 		if "/"+*item.Key == path {
// 			fmt.Println("Uploaded file found in bucket!")
// 			return
// 		}
// 	}
// 	fmt.Println(path)
// 	t.Error("Uploaded file not found in bucket! (Paths not equal)")
// 	// fmt.Println("3Found", len(resp.Contents), "items in bucket", bucket)
// 	// fmt.Println("")
// }

// func TestFileInS3(t *testing.T) {
// 	assert := assert.New(t)
// 	file1, err := ioutil.TempFile(os.TempDir(), "temp1")
// 	file2, err := ioutil.TempFile(os.TempDir(), "temp2")
// 	file3, err := ioutil.TempFile(os.TempDir(), "temp3")

// 	path1, err := filepath.Abs(file1.Name())
// 	path2, err := filepath.Abs(file2.Name())
// 	path3, err := filepath.Abs(file3.Name())

// 	err = uploadToS3(file1, path1, bucket, uploader)
// 	if err != nil {
// 		t.Error(err)
// 	}

// 	err = uploadToS3(file2, path2, bucket, uploader)
// 	if err != nil {
// 		t.Error(err)
// 	}

// 	defer os.Remove(file1.Name())
// 	defer os.Remove(file2.Name())
// 	defer os.Remove(file3.Name())

// 	in1, err := fileInS3(path1, bucket, svc)
// 	if err != nil {
// 		t.Error(err)
// 	}
// 	in2, err := fileInS3(path2, bucket, svc)
// 	if err != nil {
// 		t.Error(err)
// 	}
// 	in3, err := fileInS3(path3, bucket, svc)
// 	if err != nil {
// 		t.Error(err)
// 	}
// 	assert.Equal(in1, true)
// 	assert.Equal(in2, true)
// 	assert.Equal(in3, false)

// }
