import unittest
from mock import patch
import os
import creds


class CredsTest(unittest.TestCase):

    def test_get_iam_role(self):

        with patch.dict(os.environ, {'environment': 'production'}, clear=True):
            role, env = creds.get_iam_role()
            assert env == "production"
            assert role == "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/c2g-collector-production"

        with patch.dict(os.environ, {'environment': 'staging'}, clear=True):
            role, env = creds.get_iam_role()
            assert env == "staging"
            assert role == "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/c2g-collector-staging"

        with patch.dict(os.environ, {'environment': 'doesntexist'}, clear=True):
            try:
                role, env = creds.get_iam_role()
                raise AssertionError("should throw NameError when the given environment doesn't exist")
            except NameError:
                # This is what we expect
                pass

        with patch.dict(os.environ, {}, clear=True):
            role, env = creds.get_iam_role()
            assert role is None
            assert env is None

    def test_get_creds_from_environment(self):

        # Happy path
        with patch.dict(os.environ, {'REDSHIFT_USER': 'bob', 'REDSHIFT_PASSWORD': 'b0bzPW'}, clear=True):
            user, password = creds.get_creds_from_environment()
            assert user == 'bob'
            assert password == 'b0bzPW'

        # Sad path #1 -- no password
        with patch.dict(os.environ, {'REDSHIFT_USER': 'bob'}, clear=True):
            user, password = creds.get_creds_from_environment()
            assert user is None
            assert password is None

        # Sad path #1 -- no username
        with patch.dict(os.environ, {'REDSHIFT_PASSWORD': 'someradicalpassword'}, clear=True):
            user, password = creds.get_creds_from_environment()
            assert user is None
            assert password is None

    def test_get_redshift_creds(self):
        with patch.object(creds, 'get_iam_role',
                          return_value=("arn:aws:iam::734326455073:role/sandstorm/production/templated/role/c2g-collector-production", "production")):
            with patch('twitch.sandstorm.SandstormClient') as mock_client:
                client = mock_client.return_value
                client.GetSecret.side_effect = ["A_USER", "A_PASSWORD"]
                u, p = creds.get_redshift_creds()
                assert u == "A_USER"
                assert p == "A_PASSWORD"
