from __future__ import print_function
import json
import pg8000
import boto3
import os
import dateutil.parser
import requests
import random
import datetime

from environment import creds, db, catalog

# How many resources to batch together
# as a payload to a downstream processing lambda
batch_size = 10


def lambda_handler(event, context):
    # Get the event timestamp
    # Truncate it to the minute
    timestamp = truncate_timestamp(event['time'])
 
    # Get all accounts from catalog and construct a dict of catalog id to aws id
    accounts = construct_account_dict()
    # Get all components from catalog and construct dict of required fields
    components = get_componenets(accounts)
    random.shuffle(components)
    # Process all the resources
    process_resources(components,timestamp)


def construct_account_dict():
    url_base = catalog.get_catalog_endpoint()
    account_url = url_base + "/accounts/"
    resp = requests.get(account_url, headers={'Connection':'close'})
    if resp.status_code != 200:
        raise RuntimeError("Could not get accounts: {error}".format(error=resp.text))
    resp_accounts = resp.json()
    accounts = {}
    for account in resp_accounts:
        accounts[account['id']] = account['aws_account_id']
    return accounts

def get_componenets(accounts):
    url_base = catalog.get_catalog_endpoint()
    component_url = url_base + "/components/"
    resp = requests.get(component_url, headers={'Connection':'close'})
    if resp.status_code != 200:
        raise RuntimeError("Could not get accounts: {error}".format(error=resp.text))
    resp_components = resp.json()
    components = []
    for comp in resp_components:
        data = component_to_dict(comp)
        if data is not None:
            aws_id = accounts[comp['account']]
            data['awsaccountid']=aws_id
            components.append(data)
    return components

def component_to_dict(comp):
    label = comp['label'].split(':')
    if len(label) != 4 or comp['account'] == 0:
        return
    data = {'awsaccountname':label[1],'awsregion':label[3],'type':comp['type']}
    if comp['type'] == 'elb':
        data['elbname'] = label[2]
    elif comp['type'] == 'beanstalk':
        data['applicationname'] = label[2]
    elif comp['type'] == 'alb':
        data['albname'] = label[2]
    return data

def process_resources(resources, timestamp):
    batch = []
    for resource in resources:
        batch.append(resource)
        if len(batch) >= batch_size:
            # Flush the ELBs and reset
            flush(batch, timestamp)
            batch = []
    # Flush whatever is left
    if len(batch) != 0:
        flush(batch, timestamp)


def flush(batch, timestamp):
    data = {}
    data['timestamp'] = timestamp
    data['resources'] = batch
    try:
        env = os.environ['environment']
    except KeyError:
        return
    # Only invoke processing lambda's if this is production
    if env == 'production':
        client = boto3.client('lambda')
        client.invoke(
            FunctionName='availability-c2g-scraper-production',
            InvocationType='Event',
            Payload=json.dumps(data),
        )


def truncate_timestamp(time_string):
    timestamp = dateutil.parser.parse(time_string)
    return timestamp.replace(second=0, microsecond=0).isoformat()

# For local development
if __name__ == "__main__":
    t = datetime.datetime.utcnow().isoformat()
    lambda_handler({'time': t}, None)

