
module "tf_lambda_app" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/dta/tf_lambda_app.git?ref=env-env-var"
  aws_region = "${var.aws_region}"
  aws_profile = "twitch-central-aws"
  owner = "availability-team@twitch.tv"
  repository = "availability/c2g-collector"
  environment = "production"
  lambda_handler = "run.lambda_handler"
  lambda_function_name = "availability-c2g-collector-production"
  lambda_timeout = "240"
  lambda_description = "Triggers c2g-scraper calls to get cloudwatch metrics into graphite"
  lambda_runtime = "python2.7"
  
  lambda_policy = "${file("../../files/lambda_policy.json")}"
  principal = "007917851548"
  lambda_vpc_subnet_ids = "subnet-1a87f07e,subnet-0eb83678,subnet-828650da"
  lambda_vpc_security_group_ids = "sg-1825527e"
}

# IAM to allow access to sandstorm

resource "aws_iam_policy" "sandstorm" {
    name = "c2g-collector-production-sandstorm"
    description = "cloudwatch-to-graphite sandstorm role access"
    policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Resource": [
        "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/c2g-collector-production"
      ],
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_policy_attachment" "sandstorm-attachment" {
    name = "c2g-collector-production-sandstorm-attachment"
    roles = ["${module.tf_lambda_app.role_id}"]
    policy_arn = "${aws_iam_policy.sandstorm.arn}"
}


# Mechanism to run the lambda on a cron
# Adapted from: https://github.com/hashicorp/terraform/issues/4393#issuecomment-194287540
provider "aws" {
    alias   = "twitch-central-aws"
    profile = "twitch-central-aws"
    region  = "us-west-2"
}

resource "aws_cloudwatch_event_rule" "lambda_cron" {
    provider = "aws.twitch-central-aws"
    name = "c2g-collector-production-cron"
    description = "Fires every 1 minute"
    schedule_expression = "rate(1 minute)"
}

resource "aws_cloudwatch_event_target" "check_lambda_every_one_minute" {
    provider = "aws.twitch-central-aws"
    rule = "${aws_cloudwatch_event_rule.lambda_cron.name}"
    target_id = "c2g-collector-production-lambda"
    arn = "${module.tf_lambda_app.arn}"
}

resource "aws_lambda_permission" "allow_cloudwatch_to_call_lambda" {
    provider = "aws.twitch-central-aws"
    statement_id = "AllowExecutionFromCloudWatch"
    action = "lambda:InvokeFunction"
    function_name = "availability-c2g-collector-production"
    principal = "events.amazonaws.com"
    source_arn = "${aws_cloudwatch_event_rule.lambda_cron.arn}"
    depends_on = ["module.tf_lambda_app"]
}

# Give this Lambda permissions to invoke the downstream scraper lambda
resource "aws_iam_policy" "invoke_c2g_scraper" {
    provider = "aws.twitch-central-aws"
    name = "c2g-collector-production-invoke-scraper"
    description = "c2g-collector access to invoke downstream scraper function"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "InvokePermission",
            "Effect": "Allow",
            "Action": [
                "lambda:InvokeFunction"
            ],
            "Resource": "arn:aws:lambda:us-west-2:007917851548:function:availability-c2g-scraper-production"
        }
    ]
} 
EOF
}

resource "aws_iam_policy_attachment" "invoke_attachment" {
    provider = "aws.twitch-central-aws"
    name = "c2g-collector-production-invoke-scraper-attachment"
    roles = ["${module.tf_lambda_app.role_id}"]
    policy_arn = "${aws_iam_policy.invoke_c2g_scraper.arn}"
}
