
module "tf_lambda_app" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/dta/tf_lambda_app.git?ref=env-env-var"
  aws_region = "${var.aws_region}"
  aws_profile = "twitch-central-dev"
  owner = "availability-team@twitch.tv"
  repository = "availability/c2g-collector"
  environment = "staging"
  lambda_handler = "run.lambda_handler"
  lambda_function_name = "availability-c2g-collector-staging"
  lambda_timeout = "120"
  lambda_description = "Gathers collections of AWS resources and kicks of Cloudwatch-to-Graphite metric scraping tasks"
  lambda_runtime = "python2.7"
  
  lambda_policy = "${file("../../files/lambda_policy.json")}"
  principal = "386434559102"
  lambda_vpc_subnet_ids = "subnet-071c4763,subnet-42e99934,subnet-3e41f266"
  lambda_vpc_security_group_ids = "sg-e55f4f83"
}

# IAM to allow access to sandstorm

resource "aws_iam_policy" "sandstorm" {
    name = "c2g-collector-staging-sandstorm"
    description = "cloudwatch-to-graphite sandstorm role access"
    policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Resource": [
        "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/c2g-collector-staging"
      ],
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_policy_attachment" "sandstorm-attachment" {
    name = "c2g-collector-staging-sandstorm-attachment"
    roles = ["${module.tf_lambda_app.role_id}"]
    policy_arn = "${aws_iam_policy.sandstorm.arn}"
}


# Mechanism to run the lambda on a cron
# Adapted from: https://github.com/hashicorp/terraform/issues/4393#issuecomment-194287540
provider "aws" {
    alias   = "twitch-central-dev"
    profile = "twitch-central-dev"
    region  = "us-west-2"
}

resource "aws_cloudwatch_event_rule" "lambda_cron" {
    provider = "aws.twitch-central-dev"
    name = "c2g-collector-staging-cron"
    description = "Fires every 1 minute"
    schedule_expression = "rate(1 minute)"
}

resource "aws_cloudwatch_event_target" "check_lambda_every_one_minute" {
    provider = "aws.twitch-central-dev"
    rule = "${aws_cloudwatch_event_rule.lambda_cron.name}"
    target_id = "c2g-staging-lambda"
    arn = "${module.tf_lambda_app.arn}"
}

resource "aws_lambda_permission" "allow_cloudwatch_to_call_lambda" {
    provider = "aws.twitch-central-dev"
    statement_id = "AllowExecutionFromCloudWatch"
    action = "lambda:InvokeFunction"
    function_name = "availability-c2g-collector-staging"
    principal = "events.amazonaws.com"
    source_arn = "${aws_cloudwatch_event_rule.lambda_cron.arn}"
    depends_on = ["module.tf_lambda_app"]
}

# Give this Lambda permissions to invoke the downstream scraper lambda
resource "aws_iam_policy" "invoke_c2g_scraper" {
    provider = "aws.twitch-central-dev"
    name = "c2g-collector-staging-invoke-scraper"
    description = "c2g-collector access to invoke downstream scraper function"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "InvokePermission",
            "Effect": "Allow",
            "Action": [
                "lambda:InvokeFunction"
            ],
            "Resource": "${module.tf_lambda_app.arn}"
        }
    ]
} 
EOF
}

resource "aws_iam_policy_attachment" "invoke_attachment" {
    provider = "aws.twitch-central-dev"
    name = "c2g-collector-staging-invoke-scraper-attachment"
    roles = ["${module.tf_lambda_app.role_id}"]
    policy_arn = "${aws_iam_policy.invoke_c2g_scraper.arn}"
}
