import unittest
from mock import MagicMock
from datetime import datetime, timedelta
import run
import pytz
import json


class TestErrors(unittest.TestCase):

    def test_expose_errors(self):
        e1 = {
                'resource': {'name': 'foo'},
                'error': str(RuntimeError("something weird happened"))
            }
        e2 = {
                'resource': {'name': 'bar'},
                'error': str(NameError("name unrecognized"))
            }
        e3 = {
                'resource': {'name': 'baz'},
                'error': str(AssertionError("not what i was expecting"))
            }
        try:
            run.expose_errors([e1, e2, e3])
            raise AssertionError("expected `expose_errors` to raise a RuntimeError; it did not")
        except RuntimeError as re:
            assert str(re) == json.dumps({'errors': [e1,e2,e3]})


class TestFetchMetrics(unittest.TestCase):

    def test_fetch_metric_data_success(self):
        t_start = datetime.now()
        t_end = datetime.now()
        data = {
            'Datapoints': [
                {
                    'Average': 45,
                    'Sum': 900.56,
                    'Timestamp': t_start
                }
            ]
        }
        mock_cw = MagicMock()
        mock_cw.get_metric_statistics.return_value = data
        datapoints = run.fetch_metric_data(mock_cw, {'elbname': 'foobar', 'type': 'elb'}, t_start, t_end)
        assert len(datapoints) == 4
        for i in range(len(datapoints)):
            assert datapoints[i]['Datapoints'][0]['Sum'] == 900.56
            assert datapoints[i]['Datapoints'][0]['Average'] == 45

    def test_fetch_metric_data_synthetic_zero(self):
        t_start = datetime.now()
        t_end = datetime.now()
        data = {
            'Datapoints': []
        }
        mock_cw = MagicMock()
        mock_cw.get_metric_statistics.return_value = data
        datapoints = run.fetch_metric_data(mock_cw, {'elbname': 'foobar', 'type': 'elb'}, t_start, t_end)
        assert len(datapoints) == 4
        for i in range(len(datapoints)):
            assert datapoints[i]['Datapoints'][0]['Sum'] == float(0)
            assert datapoints[i]['Datapoints'][0]['Average'] == float(0)
            assert datapoints[i]['Datapoints'][0]['Timestamp'] == t_start


class TestTimestamps(unittest.TestCase):

    def test_format_timestamps(self):
        t = datetime.now()
        start, end = run.format_timestamps(t.isoformat())
        assert end == t
        assert start == t - timedelta(seconds=60)


class TestRecordValidation(unittest.TestCase):

    def test_validate_resource_data_success(self):
        good_elb = {
            'type': 'elb',
            'awsaccountid': 'foobar',
            'awsaccountname': 'accountname',
            'elbname': 'foobar-elb',
            'isbeanstalk': True,
            'awsregion': 'us-west-2'
        }
        good_beanstalk = {
            'type': 'beanstalk',
            'awsaccountid': 'foobar',
            'awsaccountname': 'accountname',
            'elbname': 'foobar-elb',
            'applicationname': 'cool-beanstalk-app',
            'awsregion': 'us-west-2'
        }
        good_alb = {
            'type': 'alb',
            'awsaccountid': 'foobar',
            'awsaccountname': 'accountname',
            'albname': 'foobar-alb',
            'albid': '123456abcd',
            'awsregion': 'us-west-2'
        }
        # Should just not raise any exceptions
        run.validate_resource_data(good_elb)
        run.validate_resource_data(good_beanstalk)

    def test_validate_resource_data_failure(self):
        missing_common_field = {
            'awsaccountid': 'foobar',
            'isbeanstalk': True,
            'awsregion': 'us-west-2'
        }
        try:
            run.validate_resource_data(missing_common_field)
            raise AssertionError("the resource '{r}' is not valid but passed validation".format(r=str(missing_common_field)))
        except KeyError:
            pass
        missing_field_elb = {
            'type': 'elb',
            'awsaccountid': 'foobar',
            'awsaccountname': 'accountname',
            'awsregion': 'us-west-2'
        }
        try:
            run.validate_resource_data(missing_field_elb)
            raise AssertionError("the resource '{r}' is not valid but passed validation".format(r=str(missing_field_elb)))
        except KeyError:
            pass
        missing_field_beanstalk = {
            'type': 'beanstalk',
            'awsaccountid': 'foobar',
            'awsaccountname': 'accountname',
            'elbname': 'foobar-elb',
            'awsregion': 'us-west-2'
        }
        try:
            run.validate_resource_data(missing_field_beanstalk)
            raise AssertionError("the resource '{r}' is not valid but passed validation".format(r=str(missing_field_beanstalk)))
        except KeyError:
            pass
        # No ALB ID
        missing_field_alb = {
            'type': 'alb',
            'awsaccountid': 'foobar',
            'awsaccountname': 'accountname',
            'awsregion': 'us-west-2'
        }
        try:
            run.validate_resource_data(missing_field_alb)
            raise AssertionError("the resource '{r}' is not valid but passed validation".format(r=str(missing_field_alb)))
        except KeyError:
            pass
        invalid_type = {
            'type': 'doesnotexist',
            'awsaccountid': 'foobar',
            'awsaccountname': 'accountname',
            'elbname': 'foobar-elb',
            'awsregion': 'us-west-2'
        }
        try:
            run.validate_resource_data(invalid_type)
            raise AssertionError("the resource '{r}' is not valid but passed validation".format(r=str(invalid_type)))
        except NameError:
            pass


class TestMetricPaths(unittest.TestCase):

    def test_format_metric_prefix(self):
        elb = {
            'type': 'elb',
            'awsaccountid': '12345',
            'awsaccountname': 'accountname',
            'elbname': 'awesome-elb',
            'isbeanstalk': True,
            'awsregion': 'us-west-2'
        }
        prefix = run.format_metric_prefix(elb)
        assert prefix == "cloudwatch.accountname.us-west-2.elb.awesome-elb"
        beanstalk = {
            'type': 'beanstalk',
            'awsaccountid': '4445556667',
            'awsaccountname': 'sickaccountbro',
            'elbnames': {'bs-internal-elb' : None},
            'applicationname': 'cool-beanstalk-app',
            'awsregion': 'us-west-2'
        }
        prefix = run.format_metric_prefix(beanstalk)
        assert beanstalk['elbnames']['bs-internal-elb'] == "cloudwatch.sickaccountbro.us-west-2.beanstalk.cool-beanstalk-app.bs-internal-elb"
        alb = {
            'type': 'alb',
            'awsaccountid': '4445556667',
            'awsaccountname': 'sickaccountbro',
            'albname': 'someawesomenewthing',
            'albid': '8675309',
            'awsregion': 'eu-west-1'
        }
        prefix = run.format_metric_prefix(alb)
        assert prefix == "cloudwatch.sickaccountbro.eu-west-1.alb.someawesomenewthing"

    def test_format_metric_lines(self):
        t = datetime.utcnow().replace(tzinfo=pytz.UTC)
        epoch = datetime.utcfromtimestamp(0).replace(tzinfo=pytz.UTC)
        timestamp = int((t - epoch).total_seconds())
        prefix = "arbitrary.graphite.prefix"
        data = [
            {
                'Datapoints': [
                    {
                        'Sum': 666,
                        'Timestamp': t
                    },
                ],
                'Label': 'HTTPCode_Backend_5XX'
            },
            {
                'Datapoints': [
                    {
                        'Average': 33.333,
                        'ExtendedStatistics': {
                            'p50': 50.55,
                            'p90': 90.99,
                            'p99': 99.99
                        },
                        'Timestamp': t
                    },
                ],
                'Label': 'Latency'
            }
        ]
        lines = run.format_metric_lines(prefix, data)
        assert len(lines) == 5
        assert lines[0] == "arbitrary.graphite.prefix.HTTPCode_Backend_5XX 666 {ts}".format(ts=timestamp)
        assert lines[1] == "arbitrary.graphite.prefix.Latency_average 33.333 {ts}".format(ts=timestamp)
        assert lines[2] == "arbitrary.graphite.prefix.Latency_p50 50.55 {ts}".format(ts=timestamp)
        assert lines[3] == "arbitrary.graphite.prefix.Latency_p90 90.99 {ts}".format(ts=timestamp)
        assert lines[4] == "arbitrary.graphite.prefix.Latency_p99 99.99 {ts}".format(ts=timestamp)
