#!/usr/bin/env ruby

# Read the base config, add in the cluster and memcache information

require 'yaml'
require 'net/http'
require 'resolv'

ENV_CACHE = {
    'staging' => [
        "graphite-cache-stage.xvzvze.0001.usw2.cache.amazonaws.com:11211",
        "graphite-cache-stage.xvzvze.0002.usw2.cache.amazonaws.com:11211",
        "graphite-cache-stage.xvzvze.0003.usw2.cache.amazonaws.com:11211"
    ],
    'production' => [
        "graphite-cache-prod.qwrxfk.0001.usw2.cache.amazonaws.com:11211",
        "graphite-cache-prod.qwrxfk.0002.usw2.cache.amazonaws.com:11211",
        "graphite-cache-prod.qwrxfk.0003.usw2.cache.amazonaws.com:11211"
    ]
}

output_file = ARGV[0]

# First load in the existing base config
config = YAML::load_file(File.open('conf/carbonapi-base.yaml'))

# ... and the graphite cluster definitions
clusters = YAML::load_file(File.open('conf/clusters.yaml'))

# Look for beanstalk-specific env declaration. if not found, default to
# development
if not ENV['ENV_NAME']
    # use staging
    ENVIRONMENT = 'development'
elsif ENV['ENV_NAME'].split("-").length > 1
    ENVIRONMENT = ENV['ENV_NAME'].split("-")[1]
    if ENVIRONMENT == 'prod'
        ENVIRONMENT = 'production'
    end
    ROLE = ENV['ENV_NAME'].split("-")[0]
    if ENVIRONMENT != 'staging' and ENVIRONMENT != 'production'
        puts "invalid environment '#{ENVIRONMENT}'"
        exit 1
    end
else
    puts "invalid value of ENV_NAME environment variable '#{ENV['ENV_NAME']}'"
    exit 1
end

# mix in memcache config
if ENVIRONMENT == 'development'
    config['cache']['type'] = 'mem'
else
    config['cache']['type'] = 'memcache'
    config['cache']['memcachedServers'] = ENV_CACHE[ENVIRONMENT]
end

cluster_prefixes = ['system', 'application']

# mix in cluster servers
if ENVIRONMENT == 'development'
    az = 'a'
else
    az = Net::HTTP.get(URI("http://169.254.169.254/latest/meta-data/placement/availability-zone"))[-1]
end

cluster_servers = []
cluster_prefixes.each do |p|
    clusters['twitch_graphite_clusters'][ENVIRONMENT]["#{p}_#{az}"].each do |i, host|
        cluster_servers << "http://#{Resolv.getaddress(host)}:80"
    end
end
config['upstreams']['backends'] = cluster_servers

# Set up metrics for staging and production
if ENVIRONMENT == 'staging' or ENVIRONMENT == 'production'
    config['graphite']['host'] = 'graphite.internal.justin.tv:2003'
    config['graphite']['pattern'] = "carbon.#{ENVIRONMENT}.carbonapi.#{ROLE}.{fqdn}"
end

f = File.open(output_file, 'w')
f.write(YAML::dump(config))