package cw2graphite

import (
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/aws/aws-sdk-go/service/cloudwatch/cloudwatchiface"
	"github.com/aws/aws-sdk-go/service/sts"

	log "github.com/Sirupsen/logrus"
	graphite "github.com/marpaia/graphite-golang"
)

// C2GMetric encodes information about a CloudWatch metric
type C2GMetric struct {
	Namespace  *string
	MetricName *string
	Statistics []*string
	Dimensions []*cloudwatch.Dimension
	Prefix     string
}

// C2GTask represents a periodic request for CloudWatch data
type C2GTask struct {
	Cw       cloudwatchiface.CloudWatchAPI
	Period   time.Duration
	Graphite *graphite.Graphite
	Metrics  []*C2GMetric
}

// fire makes request for CloudWatch data, and sends the resulting data into
// graphite
func (c2g *C2GTask) fire() {
	start := time.Now().Add(-1 * time.Second * 60)
	end := time.Now()
	for _, m := range c2g.Metrics {
		period := aws.Int64(int64(c2g.Period.Seconds()))
		input := &cloudwatch.GetMetricStatisticsInput{
			Namespace:  m.Namespace,
			MetricName: m.MetricName,
			Period:     period,
			StartTime:  &start,
			EndTime:    &end,
			Statistics: m.Statistics,
			Dimensions: m.Dimensions,
		}

		output, err := c2g.Cw.GetMetricStatistics(input)
		if err != nil {
			log.Warnf("Failed Cloudwatch Namespace %s, %s, %v, for time period: %s : %s",
				c2g.Metrics[0].Namespace, c2g.Metrics[0].MetricName,
				c2g.Metrics[0].Dimensions, start.String(), end.String())
			return
		}

		for _, d := range output.Datapoints {
			reportStat(d, c2g.Graphite, m.Prefix, *m.Statistics[0])
		}
	}
}

func reportStat(datapoint *cloudwatch.Datapoint, g *graphite.Graphite, prefix string, stat string) error {
	var val string // Graphite wants a string for a value...
	switch stat {
	case "Sum":
		val = fmt.Sprintf("%f", *datapoint.Sum)
	case "Average":
		val = fmt.Sprintf("%f", *datapoint.Average)
	}

	m := graphite.Metric{
		Name:      prefix,
		Value:     val,
		Timestamp: datapoint.Timestamp.Unix(),
	}

	log.Debugf("Graphite metric {path=%s,value=%s,timestamp=%d}", m.Name, m.Value, int64(m.Timestamp))

	reconnected := false
	for {
		if err := g.SendMetric(m); err != nil {
			// Try reconnecting and running again.
			if reconnected {
				// If we already retried then give up and return the error
				return err
			} else {
				// First try, attempt to reconnect and resend the metric.
				g.Connect()
				reconnected = true
			}
		} else {
			// Success! No error!
			return nil
		}
	}
	// Reconnect on
	return nil
}

func trackResource(accountNumber, accountAlias, awsRegion, roleArn string, metrics []*C2GMetric, graphiteHost string, graphitePort int, stop chan bool) {
	// TODO: Setup input args
	// TODO: Simplify this method, extract components
	// TODO: Implement forever-loop logic

	// Set up credentials for this ELB
	sess := session.New()
	conf := &aws.Config{Region: aws.String(awsRegion)}
	cred := credentials.NewCredentials(
		&stscreds.AssumeRoleProvider{
			RoleARN: roleArn,
			Client:  sts.New(sess),
		},
	)
	_, err := cred.Get()
	if err != nil {
		log.Error(err)
	}
	conf = conf.WithCredentials(cred)

	// Instantiate a session
	cw := cloudwatch.New(sess, conf)

	// Establish connection to graphite
	g, err := graphite.NewGraphite(graphiteHost, graphitePort)

	if err != nil {
		log.Error(err)
	}

	r := &C2GTask{
		Cw:       cw,
		Graphite: g,
		Period:   60 * time.Second,
		Metrics:  metrics,
	}

	// Infinite loop to get metrics, until a stop signal is received
	ticker := time.NewTicker(time.Second * 60)
	for {
		select {
		case <-ticker.C:
			for _, m := range metrics {
				log.Debugf("Fetching metric {account=%s,region=%s,metric=%s}", accountAlias, awsRegion, m.Prefix)
			}
			r.fire()
		case <-stop:
			for _, m := range metrics {
				log.Debugf("Stopping resource scraping {account=%s,region=%s,metric=%s}", accountAlias, awsRegion, m.Prefix)
			}
			return
		}
	}
}

// TrackELB submits CloudWatch metrics for an ELB into graphite
// Spins off a go routine and hands back a channel to send into if the go routine should stop
func TrackELB(accountNumber, accountAlias, elbName, awsRegion, roleArn string, graphiteHost string, graphitePort int) chan bool {
	metrics := getELBMetricInformation(accountAlias, elbName, awsRegion)
	c := makeStopChannel()
	go trackResource(accountNumber, accountAlias, awsRegion, roleArn, metrics, graphiteHost, graphitePort, c)
	return c
}

// TrackBeanstalk submits CloudWatch metrics for a beanstalk-backed ELB into graphite
func TrackBeanstalk(accountNumber, accountAlias, applicationName, elbName, awsRegion, roleArn string, graphiteHost string, graphitePort int) chan bool {
	metrics := getBeanstalkMetricInformation(accountAlias, applicationName, elbName, awsRegion)
	c := makeStopChannel()
	go trackResource(accountNumber, accountAlias, awsRegion, roleArn, metrics, graphiteHost, graphitePort, c)
	return c
}

// For the time being this is the same as ELB metric info, just in a different bucket
func getBeanstalkMetricInformation(accountAlias, applicationName, elbName, awsRegion string) []*C2GMetric {
	bucket := fmt.Sprintf("cloudwatch.%s.beanstalk.%s.%s", accountAlias, applicationName, elbName)
	sum := []*string{aws.String("Sum")}
	avg := []*string{aws.String("Average")}
	dims := []*cloudwatch.Dimension{
		&cloudwatch.Dimension{
			Name:  aws.String("LoadBalancerName"),
			Value: aws.String(elbName),
		},
	}
	m := []*C2GMetric{
		&C2GMetric{
			Namespace:  aws.String("AWS/ELB"),
			MetricName: aws.String("RequestCount"),
			Statistics: sum,
			Dimensions: dims,
			Prefix:     fmt.Sprintf("%s.request_count", bucket),
		},
		&C2GMetric{
			Namespace:  aws.String("AWS/ELB"),
			MetricName: aws.String("HTTPCode_Backend_5XX"),
			Statistics: sum,
			Dimensions: dims,
			Prefix:     fmt.Sprintf("%s.httpcode_backend_5xx", bucket),
		},
		&C2GMetric{
			Namespace:  aws.String("AWS/ELB"),
			MetricName: aws.String("Latency"),
			Statistics: avg,
			Dimensions: dims,
			Prefix:     fmt.Sprintf("%s.latency", bucket),
		},
	}
	return m
}

// We want request count, 5XX count, and latency
func getELBMetricInformation(accountAlias, elbName, awsRegion string) []*C2GMetric {
	bucket := fmt.Sprintf("cloudwatch.%s.elb.%s", accountAlias, elbName)
	sum := []*string{aws.String("Sum")}
	avg := []*string{aws.String("Average")}
	dims := []*cloudwatch.Dimension{
		&cloudwatch.Dimension{
			Name:  aws.String("LoadBalancerName"),
			Value: aws.String(elbName),
		},
	}
	m := []*C2GMetric{
		&C2GMetric{
			Namespace:  aws.String("AWS/ELB"),
			MetricName: aws.String("RequestCount"),
			Statistics: sum,
			Dimensions: dims,
			Prefix:     fmt.Sprintf("%s.request_count", bucket),
		},
		&C2GMetric{
			Namespace:  aws.String("AWS/ELB"),
			MetricName: aws.String("HTTPCode_Backend_5XX"),
			Statistics: sum,
			Dimensions: dims,
			Prefix:     fmt.Sprintf("%s.httpcode_backend_5xx", bucket),
		},
		&C2GMetric{
			Namespace:  aws.String("AWS/ELB"),
			MetricName: aws.String("Latency"),
			Statistics: avg,
			Dimensions: dims,
			Prefix:     fmt.Sprintf("%s.latency", bucket),
		},
	}
	return m
}

func makeStopChannel() chan bool {
	return make(chan bool, 1)
}
