package main

import (
	"context"
	"bytes"
	"code.justin.tv/qe/twitchldap"
	"encoding/json"
	"io/ioutil"
	"log"
	"net/http"
	"time"
	"code.justin.tv/qe/contacts-service/rpc/contacts"
	"code.justin.tv/availability/goracle/messaging"
	"code.justin.tv/availability/goracle/config"
	"fmt"
	"github.com/nlopes/slack"
	"os"
	"strconv"
	"strings"
	"code.justin.tv/availability/goracle/emailreport"
	"github.com/aws/aws-lambda-go/lambda"
)
const (
	contactsServiceUrl = "https://contacts-api.internal.justin.tv"
	listServiceNamesGraphql = `{"operationName":"FetchAllServices", "variables":{}, "query":"query FetchAllServices {services{id,name,primary_owner{uid}}}"}`
	slackMessageBody = `Hello %s, as the manager of %s who is the primary owner of %s but is no longer active in Twitch LDAP.

Please take one of these actions:

1) Assign to self

2) Assign to someone else

If the inactive status of %s is incorrect, please contact #service-catalog on slack
`
)
var (

)

type Services struct {
	Data struct {
		Services []struct {
			ID string `json:"id"`
			Name string `json:"name"`
			PrimaryOwner struct {
				UID string `json:"uid"`
			} `json:"primary_owner"`
		} `json:"services"`
	} `json:"data"`
}

func listServices() (*Services, error) {
	serviceCatUrl := config.ServiceCatalogAPIURL()
	debugOverrideEnvUrl := os.Getenv("DEBUG_OVERRIDE_SERVICECAT_URL")
	if debugOverrideEnvUrl != "" { // debug override for testing
		log.Printf("in debug mode. overriding URL to %s", debugOverrideEnvUrl)
		serviceCatUrl = debugOverrideEnvUrl
	}
	graphqlEndpoint := fmt.Sprintf("%s/api/v2/query", serviceCatUrl)
	var payload = []byte(listServiceNamesGraphql)
	req, err := http.NewRequest("POST", graphqlEndpoint, bytes.NewBuffer(payload))
	if err != nil {
		return nil, err
	}

	client := &http.Client{}
	resp, err := client.Do(req)
	defer resp.Body.Close()
	if err != nil {
		return nil, err
	}

	bodyBytes, _ := ioutil.ReadAll(resp.Body)
	var services Services
	err = json.Unmarshal(bodyBytes, &services)
	if err != nil {
		return nil, err
	}

	return &services, err
}

// LambdaHandler is the function that AWS lambda invokes at a predefined schedule (daily)
// To test locally or in AWS env, set these overrides:
// DEBUG_OVERRIDE_MANAGER : manager uid
// DEBUG_OVERRIDE_ENV_URL : service catalog url
// DEBUG_OVERRIDE_LOOK_BACK_HOURS : override look back hours
func LambdaHandler() {
	// 1) graphql query to get all services
	services, err := listServices()
	if err != nil {
		log.Fatalf("ERROR! unable to query from service catalog: %s", err.Error())
	}

	ldapClient, err := twitchldap.NewClient()
	defer ldapClient.Close()
	if err != nil { // fatal exit if ldap client is not working. it is required for successful processing
		log.Fatalf("ERROR! failed creating ldap client: %s", err.Error())
	}

	err = config.ParseConfig("config.toml")
	if err != nil {
		log.Printf("Warning - Failed loading config.toml: %s", err.Error())
	}
	debugOverrideManager := os.Getenv("DEBUG_OVERRIDE_MANAGER")
	debugOverrideEnvUrl := os.Getenv("DEBUG_OVERRIDE_SERVICECAT_URL")
	debugLookBackHours := os.Getenv("DEBUG_OVERRIDE_LOOK_BACK_HOURS")
	slackApiTokenOverride := os.Getenv("SLACK_API_TOKEN")
	if slackApiTokenOverride != "" {
		log.Printf("overriding slack token using env var SLACK_API_TOKEN")
		config.Config.SlackToken = slackApiTokenOverride
	}

	serviceCatUrl := config.ServiceCatalogURL()
	if debugOverrideEnvUrl != "" { // debug override for testing
		log.Printf("in debug mode. overriding URL to %s", debugOverrideEnvUrl)
		serviceCatUrl = debugOverrideEnvUrl
	}
	lookBackwardHours := 24 // how many hours to look back to detect inactive users
	if debugLookBackHours != "" { // debug override for testing
		log.Printf("in debug mode. overriding look back hours to %s", debugLookBackHours)
		lookBackwardHours, err = strconv.Atoi(debugLookBackHours)
		if err != nil {
			log.Printf("env var DEBUG_OVERRIDE_LOOK_BACK_HOURS has invalid value %s", debugLookBackHours)
		}
	}
	for _, service := range services.Data.Services {
		// 2) for each service owner, make ldap query for inactive
		log.Printf("service: id=%s, name=%s", service.ID, service.Name)
		log.Printf("\towner: %s", service.PrimaryOwner.UID)
		if service.PrimaryOwner.UID == "" {
			log.Printf("\t\twarning: service has NO OWNER")
			continue
		}
		ldapUser, err := ldapClient.GetUserInfoByName(service.PrimaryOwner.UID)
		if err != nil {
			log.Printf("ERROR! problem getting ldapUser %s: %s ", service.PrimaryOwner.UID, err.Error())
			continue
		}
		log.Printf("\t\tldapUser=%s, inactive=%s, inactive=%f hours. lookback threshold to inactive trigger notification=%d hrs", ldapUser.UID, strconv.FormatBool(ldapUser.Inactive), time.Since(ldapUser.InactiveTime).Hours(),lookBackwardHours)
		if ldapUser.Inactive && time.Since(ldapUser.InactiveTime).Hours() < float64(lookBackwardHours) {
			// 3) for each inactive owner inactivated since the last lambda run, notify manager via slack
			currentPrimaryOwnerUserId := ldapUser.UID
			api := contacts.NewContactsProtobufClient(contactsServiceUrl, &http.Client{})
			managerResult, err := api.GetManager(context.Background(), &contacts.UserRequest{UserId: currentPrimaryOwnerUserId})
			if err != nil {
				log.Printf("ERROR! problem querying %s's manager: %s", ldapUser.UID, err.Error())
			}

			// resolve all the variables
			managerUserId := ""
			managerUserName := ""
			managerEmpId := uint32(0)
			serviceId := service.ID
			if managerResult == nil || managerResult.Person == nil {
				// ldapUser has already been deleted in salesforce backend, so contacts service can't provide manager info
				// there should be a pagerduty alarm for this case for manual resolution. it cannot be resolved programatically
				if debugOverrideManager != "" { // debug override for testing
					log.Printf("in debug mode. overriding manager references to %s", debugOverrideManager)
					managerOverride, err := api.GetPerson(context.Background(), &contacts.UserRequest{UserId: debugOverrideManager})
					if err != nil {
						log.Printf("ERROR! problem querying manager override %s: %s", debugOverrideManager, err.Error())
						continue
					}
					managerUserId = managerOverride.Person.UserId
					managerUserName = managerOverride.Person.PreferredName
					managerEmpId = managerOverride.Person.EmployeeNumber
				} else {
					log.Printf("ERROR! ldapUser %s has been deleted in salesforce backend, so the manager cannot be deteremined for service %s", ldapUser.UID, service.Name)
					continue
				}
			} else {
				managerUserId = managerResult.Person.UserId
				managerUserName = managerResult.Person.PreferredName
				managerEmpId = managerResult.Person.EmployeeNumber
			}

			// prepare slack message
			slackMsgBody := fmt.Sprintf(slackMessageBody, managerUserName, currentPrimaryOwnerUserId, service.Name, currentPrimaryOwnerUserId)
			assignTo := &slack.AttachmentAction{
				Name:  "Assign",
				Text:  "Assign",
				Type:  "button",
				Style: "primary",
				URL:   fmt.Sprintf("%s/services/%s?edit=1", serviceCatUrl, serviceId),
			}
			assignToMe := &slack.AttachmentAction{
				Name:  "AssignToMe",
				Text:  "Assign To Me",
				Type:  "button",
				Style: "primary",
				URL:   fmt.Sprintf("%s/messaging/assignOwner/?serviceID=%s&ownerEmployeeId=%d", serviceCatUrl, serviceId, managerEmpId),
			}
			attachmentButtons := []slack.Attachment{
				{
					Actions:    []slack.AttachmentAction{*assignToMe, *assignTo},
				},
			}
			// send out slack message
			log.Printf("attempting to send slack notification to %s", managerUserName)
			err = messaging.SlackPostDM(managerUserId, managerUserName, "Primary Ownership Transfer", slackMsgBody, service.Name, service.ID, true, attachmentButtons)
			if err != nil || os.Getenv("DEBUG_SEND_EMAIL") == "true" {
				if err != nil {
					log.Printf("ERROR! failed to send slack message to new primary owner: %s", err)
				}
				if os.Getenv("DEBUG_SEND_EMAIL") == "true" {
					sendEmailNotification(managerUserId, []string{currentPrimaryOwnerUserId},  "Primary Ownership Transfer", slackMsgBody)
				}
			}
		}
	}
}

func sendEmailNotification(recipientUID string, ccUIDs []string, subject string, body string) {
	log.Printf("attempting to send email notification to %s", recipientUID)
	ldapClient, err := twitchldap.NewClient()
	if err != nil {
		log.Printf("ERROR! failed to initialize ldap client: %s", err.Error())
		return
	}
	userInfo, err := ldapClient.GetUserInfoByName(recipientUID)
	if err != nil {
		log.Printf("ERROR! failed to get user info from ldap client for %s: %s", recipientUID, err.Error())
		return
	}
	log.Printf("sending email to %s", userInfo.Mail)
	htmlBody := strings.Replace(body,"\n", "<br/>", -1)
	htmlBody += fmt.Sprintf("<br/><br/>Click <a href='%s/serviceOwner/%s'>here</a> to see your services.",
		config.ServiceCatalogURL(), recipientUID)
	emailreport.SendMail("goracle-noreply@justin.tv", []string{userInfo.Mail}, nil, subject, htmlBody)
}

func main() {
	lambda.Start(LambdaHandler)
}

