resource "aws_cloudwatch_event_rule" "run_daily_inactive_owners_development" {
  name = "${var.lambda_function_name}-daily"
  schedule_expression = "cron(55 7 * * ? *)"
}

resource "aws_cloudwatch_event_target" "lambda_event_target_staging" {
  rule = "${aws_cloudwatch_event_rule.run_daily_inactive_owners_development.name}"
  arn = "${aws_lambda_function.tf_lambda_function.arn}"
}

resource "aws_lambda_permission" "allow_cloudwatch_inactive_owners_development" {
  statement_id = "AllowExecutionFromCloudWatch_inactive_owners_development"
  action = "lambda:InvokeFunction"
  function_name = "${var.lambda_function_name}"
  principal = "events.amazonaws.com"
  source_arn = "${aws_cloudwatch_event_rule.run_daily_inactive_owners_development.arn}"
}

resource "aws_iam_role_policy" "tf_lambda_policy" {
  name = "lambda_policy-${var.lambda_function_name}"
  role = "${aws_iam_role.tf_lambda_uploader_role.id}"
  policy = "${file("files/lambda_policy.json")}"
}

resource "aws_iam_role" "tf_lambda_uploader_role" {
  provider = "aws"
  name = "iam_for_tw_lambda_uploader-${var.lambda_function_name}"
  assume_role_policy = "${var.lambda_assume_role_policy}"
}

resource "aws_lambda_permission" "allow_api_gateway-invoke_function" {
  statement_id = "AllowLambdaInvokeFunctionFor${var.lambda_function_name}FromAPIGateway"
  action = "lambda:InvokeFunction"
  function_name = "${aws_lambda_function.tf_lambda_function.arn}"
  principal = "apigateway.amazonaws.com"
}

resource "aws_lambda_permission" "allow_tw_lambda_uploader-get_function" {
  statement_id = "AllowLambdaGetFunctionFor${var.lambda_function_name}FromUploader"
  action = "lambda:GetFunction"
  function_name = "${aws_lambda_function.tf_lambda_function.arn}"
  principal = "${var.principal}"
  source_account = "${var.external_uploader_account}"
  source_arn = "arn:aws:iam:${var.external_uploader_region}:${var.external_uploader_account}:user/${var.external_uploader_user}"
}

resource "aws_lambda_permission" "allow_tw_lambda_uploader-update_function_code" {
  statement_id = "AllowLambdaUpdateFunctionCodeFor${var.lambda_function_name}FromUploader"
  action = "lambda:UpdateFunctionCode"
  function_name = "${aws_lambda_function.tf_lambda_function.arn}"
  principal = "${var.principal}"
  source_account = "${var.external_uploader_account}"
  source_arn = "arn:aws:iam:${var.external_uploader_region}:${var.external_uploader_account}:user/${var.external_uploader_user}"
}

provider "aws" {
  region = "us-west-2"
}

data "aws_s3_bucket_object" "lambda" {
  bucket = "${aws_s3_bucket.availability_goracle_job_inactive_owners.bucket}"
  key = "${var.environment}"
}

resource "aws_lambda_function" "tf_lambda_function" {
  provider = "aws"
  function_name = "${var.lambda_function_name}"
  role = "${aws_iam_role.tf_lambda_uploader_role.arn}"
  description = "Identifies inactive primary owner and notifies managers"
  s3_bucket = "${data.aws_s3_bucket_object.lambda.bucket}"
  s3_key = "${data.aws_s3_bucket_object.lambda.key}"
  handler = "main"
  timeout = "60"
  runtime = "${var.lambda_runtime}"
  memory_size = "128"

  vpc_config {
    subnet_ids = [
      "subnet-071c4763",
      "subnet-42e99934",
      "subnet-3e41f266"]
    security_group_ids = [
      "sg-a52e92d5",
      "sg-fb2c908b",
      "sg-8209f8f8",
      "sg-a305f4d9"]
  }

  environment {
    variables {
      ENVIRONMENT = "${var.environment}"
      DEBUG_OVERRIDE_SERVICECAT_URL = "https://status-development.internal.justin.tv"
      DEBUG_OVERRIDE_MANAGER = "lukemng"
      SLACK_API_TOKEN = "${var.slack_token}"
      DEBUG_SEND_EMAIL = "true"
    }
  }

  lifecycle {
    create_before_destroy = true
  }

}

