resource "aws_cloudwatch_event_rule" "run_daily_inactive_owners_production" {
  name = "${var.lambda_function_name}-daily"
  schedule_expression = "cron(55 7 * * ? *)"
}

resource "aws_cloudwatch_event_target" "lambda_event_target_staging" {
  rule = "${aws_cloudwatch_event_rule.run_daily_inactive_owners_production.name}"
  arn = "${aws_lambda_function.tf_lambda_function.arn}"
}

resource "aws_lambda_permission" "allow_cloudwatch_inactive_owners_production" {
  statement_id = "AllowExecutionFromCloudWatch_inactive_owners_production"
  action = "lambda:InvokeFunction"
  function_name = "${var.lambda_function_name}"
  principal = "events.amazonaws.com"
  source_arn = "${aws_cloudwatch_event_rule.run_daily_inactive_owners_production.arn}"
}

resource "aws_iam_role_policy" "tf_lambda_policy" {
  name = "lambda_policy-${var.lambda_function_name}"
  role = "${aws_iam_role.tf_lambda_uploader_role.id}"
  policy = "${file("files/lambda_policy.json")}"
}

resource "aws_iam_role" "tf_lambda_uploader_role" {
  provider = "aws"
  name = "iam_for_tw_lambda_uploader-${var.lambda_function_name}"
  assume_role_policy = "${var.lambda_assume_role_policy}"
}

resource "aws_lambda_permission" "allow_api_gateway-invoke_function" {
  statement_id = "AllowLambdaInvokeFunctionFor${var.lambda_function_name}FromAPIGateway"
  action = "lambda:InvokeFunction"
  function_name = "${aws_lambda_function.tf_lambda_function.arn}"
  principal = "apigateway.amazonaws.com"
}

resource "aws_lambda_permission" "allow_tw_lambda_uploader-get_function" {
  statement_id = "AllowLambdaGetFunctionFor${var.lambda_function_name}FromUploader"
  action = "lambda:GetFunction"
  function_name = "${aws_lambda_function.tf_lambda_function.arn}"
  principal = "${var.principal}"
  source_account = "${var.external_uploader_account}"
  source_arn = "arn:aws:iam:${var.external_uploader_region}:${var.external_uploader_account}:user/${var.external_uploader_user}"
}

resource "aws_lambda_permission" "allow_tw_lambda_uploader-update_function_code" {
  statement_id = "AllowLambdaUpdateFunctionCodeFor${var.lambda_function_name}FromUploader"
  action = "lambda:UpdateFunctionCode"
  function_name = "${aws_lambda_function.tf_lambda_function.arn}"
  principal = "${var.principal}"
  source_account = "${var.external_uploader_account}"
  source_arn = "arn:aws:iam:${var.external_uploader_region}:${var.external_uploader_account}:user/${var.external_uploader_user}"
}

provider "aws" {
  region = "us-west-2"
}

data "aws_s3_bucket_object" "lambda" {
  bucket = "${var.lambda_bucket}"
  key = "${var.environment}"
}

resource "aws_lambda_function" "tf_lambda_function" {
  provider = "aws"
  function_name = "${var.lambda_function_name}"
  role = "${aws_iam_role.tf_lambda_uploader_role.arn}"
  description = "Identifies inactive primary owner and notifies managers"
  s3_bucket = "${data.aws_s3_bucket_object.lambda.bucket}"
  s3_key = "${data.aws_s3_bucket_object.lambda.key}"
  handler = "main"
  timeout = "60"
  runtime = "${var.lambda_runtime}"
  memory_size = "128"

  vpc_config {
    subnet_ids = [
      "subnet-1a87f07e",
      "subnet-0eb83678",
      "subnet-828650da"]
    security_group_ids = [
      "sg-235e3b53",
      "sg-f3563383",
      "sg-df17f3a1",
      "sg-ea09ed94"]
  }

  environment {
    variables {
      ENVIRONMENT = "${var.environment}"
      SLACK_API_TOKEN = "${var.slack_token}"
    }
  }

  lifecycle {
    create_before_destroy = true
  }

}

