package catalog

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

    "code.justin.tv/qe/twitchldap"

	"github.com/jinzhu/gorm"
	"github.com/sirupsen/logrus"
	"code.justin.tv/availability/goracle/goracleUser"
)

//TODO: review this method vs a lookup table in the database
const (
	LogTypeService           = "service"
	LogTypeComponent         = "component"
	LogTypeMetric            = "metric"
	LogTypeQuery             = "query"
	LogTypeAttribute         = "attribute"
	LogTypeFeature           = "feature"
	LogTypeServiceDependency = "service_dependency"
	LogTypeServiceState      = "state"

	LogOpCreate = "create"
	LogOpUpdate = "update"
	LogOpDelete = "delete"
)

// LogRecord is used for audit log trails
type LogRecord struct {
	gorm.Model
	Action    string
	ItemType  string
	ItemLabel string
	ItemID    uint
	Before    string `gorm:"size:4000"`
	After     string `gorm:"size:4000"`
	Author    string
}

// WithBefore sets the Before field of a LogRecord
func (l *LogRecord) WithBefore(r interface{}) *LogRecord {
	j, err := json.Marshal(r)
	if err != nil {
		logrus.Errorf("Failed to encode json for LogRecord.WithBefore: %s", err)
		return l
	}
	l.Before = string(j)
	return l

}

// WithAfter sets the After field of a LogRecord
func (l *LogRecord) WithAfter(r interface{}) *LogRecord {
	j, err := json.Marshal(r)
	if err != nil {
		logrus.Errorf("Failed to encode json for LogRecord.WithBefore: %s", err)
		return l
	}
	l.After = string(j)
	return l
}

type LogInfo struct {
	Type  string
	ID    uint
	Label string
	Data  string
}

type Loggable interface {
	LogInfo() *LogInfo
}

func NewLogRecordFromLogInfos(action string, bl, al *LogInfo, author string) *LogRecord {
	lr := &LogRecord{
		Action: action,
		Author: author,
	}
	if bl == nil && al == nil {
		logrus.Warningf("NewLogRecordFromLogInfo received nil for both before and after parameters, this will result in an unhelpful log entry: %+v", lr)
	}
	if bl != nil {
		lr.ItemType = bl.Type
		lr.ItemID = bl.ID
		lr.ItemLabel = bl.Label
		lr.Before = bl.Data
	}
	if al != nil {
		// when both before and after are passed the after fields take precedence
		lr.ItemType = al.Type
		lr.ItemID = al.ID
		lr.ItemLabel = al.Label
		lr.After = al.Data
	}
	return lr
}

type serviceDep struct {
	ID   uint   `json:"id"`
	Name string `json:"name"`
}
type ServiceLog struct {
	ID                    uint         `json:"id"`
	Name                  string       `json:"name"` // Globally unique human service name
	PrimaryOwnerName      string       `json:"primary_owner_name"`
	PrimaryOwnerUID       string       `json:"primary_owner_uid"`
	Description           string       `json:"description"`
	Known                 bool         `json:"known"`
	Type                  string       `json:"type"`
	Environment           string       `json:"environment"`
	Components            []uint       `json:"components"`
	AvailabilityObjective float64      `json:"availability_objective,string"`
	CreatedAt             time.Time    `json:"created"`
	UpdatedAt             time.Time    `json:"updated"`
	Attributes            []string     `json:"attributes"`
	Downstreams           []serviceDep `json:"downstreams"`
	Upstreams             []serviceDep `json:"upstreams"`
	PagerDuty             string       `json:"pagerduty"`
	Slack                 string       `json:"slack"`
	State                 string       `json:"state"`
}

func (s *Service) LogInfo() *LogInfo {
	// Get LDAP info about the primary owner
	// These will get overwritten unless there is an error
	primaryOwnerName := ""
	primaryOwnerUID := ""
	if s.PrimaryOwnerID != 0 {
        c, err := twitchldap.NewClient()
		defer c.Close()
		if err == nil {
			ui, err := c.GetUserInfo(uint32(s.PrimaryOwnerID))
			if err != nil {
				logrus.Warnf("error fetching LDAP data for service changelog: %s", err.Error())
			} else {
				primaryOwnerName = ui.CN
				primaryOwnerUID = ui.UID
			}
		}
	}
	l := &ServiceLog{
		ID:                    s.ID,
		Name:                  s.Name,
		PrimaryOwnerName:      primaryOwnerName,
		PrimaryOwnerUID:       primaryOwnerUID,
		Description:           s.Description,
		PagerDuty:             s.PagerDuty,
		Slack:                 s.Slack,
		Known:                 s.Known,
		CreatedAt:             s.CreatedAt,
		UpdatedAt:             s.UpdatedAt,
		AvailabilityObjective: s.AvailabilityObjective,
		Environment:           s.Environment,
		Downstreams:           getDeps(s.ID, false),
		Upstreams:             getDeps(s.ID, true),
		State:                 s.State.Name,
	}
	if s.ServiceType != nil {
		l.Type = s.ServiceType.Label
	}
	compIds := make([]uint, len(s.Components))
	for i, comp := range s.Components {
		compIds[i] = comp.ID
	}
	l.Components = compIds

	params := map[string]interface{}{
		"object_type": LogTypeService,
		"object_id":   s.ID,
	}
	attrs, err := GetCatalog().GetAttributes(params)
	if err != nil {
		logrus.Errorf("GetAttributes failed for service %d", s.ID)
	}
	l.Attributes = make([]string, len(attrs))
	for i, a := range attrs {
		if a == nil {
			logrus.Errorf("Got nil attribute for service %d", s.ID)
			continue
		}
		attrStr := fmt.Sprintf("%s:%s", a.Name, a.Value)
		l.Attributes[i] = attrStr
	}

	j, err := json.Marshal(l)
	if err != nil {
		logrus.Errorf("Failed to encode json for Service.LogInfo: %s", err)
	}
	return &LogInfo{
		Type:  LogTypeService,
		ID:    l.ID,
		Label: l.Name,
		Data:  string(j),
	}
}

func getDeps(sid uint, upstream bool) []serviceDep {
	params := make(map[string]interface{})
	if upstream {
		params["downstream_service_id"] = sid
	} else {
		params["root_service_id"] = sid
	}
	deps, err := GetCatalog().GetServiceDependencies(params)
	if err != nil {
		logrus.Errorf("GetServiceDependencies failed for service %d", sid)
	}
	var ids []serviceDep
	for _, dep := range deps {
		if dep == nil {
			logrus.Errorf("Got nil service dependency for %d", sid)
			continue
		}
		var depid uint
		if upstream {
			depid = dep.RootServiceID
		} else {
			depid = dep.DownstreamServiceID
		}
		depSvc, err := GetCatalog().GetServiceByID(depid)
		if err != nil {
			logrus.Errorf("GetServiceByID failed for %d: %s", sid, err)
		}
		ids = append(ids, serviceDep{ID: depid, Name: depSvc.Name})
	}
	return ids
}

type attributeLog struct {
	ID         uint   `json:"id"`
	Name       string `json:"name"`
	Value      string `json:"value"`
	ObjectType string `json:"object_type"`
	ObjectID   uint   `json:"object_id"`
}

func (a *Attribute) LogInfo() *LogInfo {
	l := &attributeLog{
		ID:         a.ID,
		Name:       a.Name,
		ObjectType: a.ObjectType,
		ObjectID:   a.ObjectId,
		Value:      a.Value,
	}
	j, err := json.Marshal(l)
	if err != nil {
		logrus.Errorf("Failed to encode json for Service.LogInfo: %s", err)
	}
	return &LogInfo{
		Type:  LogTypeAttribute,
		ID:    l.ID,
		Label: l.Name,
		Data:  string(j),
	}
}

type componentLog struct {
	ID          uint   `json:"id"`
	Label       string `json:"label"`
	Claimed     bool   `json:"claimed"`
	Name        string `json:"name"`
	Description string `json:"description"`
	Type        string `json:"type"`
	Metrics     []uint `json:"metric_ids"`
	Rollup      bool   `json:"rollup"`
	ServiceID   uint   `json:"service_id"`
}

func (dc *Component) LogInfo() *LogInfo {
	cl := &componentLog{
		ID:          dc.ID,
		Type:        dc.Type,
		Label:       dc.Label,
		Claimed:     dc.Claimed,
		Name:        dc.Name,
		Description: dc.Description,
		Rollup:      dc.Rollup,
		ServiceID:   dc.ServiceID,
	}
	metIds := make([]uint, 0)
	for _, met := range dc.Metrics {
		metIds = append(metIds, met.ID)
	}
	cl.Metrics = metIds

	j, err := json.Marshal(cl)
	if err != nil {
		logrus.Errorf("Failed to encode json for Service.LogInfo: %s", err)
	}
	return &LogInfo{
		Type:  LogTypeComponent,
		ID:    cl.ID,
		Label: cl.Name,
		Data:  string(j),
	}
}

type metricLog struct {
	ID               uint    `json:"id"`
	Label            string  `json:"label"`
	Name             string  `json:"name"`
	Description      string  `json:"description"`
	AutoGenerated    bool    `json:"autogenerated"`
	Queries          []uint  `json:"queries"`
	Threshold        float64 `json:"threshold"`
	CalculationType  string  `json:"calculation_type"`
	ComponentRollup  bool    `json:"component_rollup"`
	ComponentID      uint    `json:"component_id"`
	LatencyQuery     string  `json:"latency_query"`
	LatencyObjective float64 `json:"latency_objective"`
}

func (dm *Metric) LogInfo() *LogInfo {
	ml := &metricLog{
		ID:               dm.ID,
		Label:            dm.Label,
		Name:             dm.Name,
		Description:      dm.Description,
		ComponentRollup:  dm.ComponentRollup,
		AutoGenerated:    dm.AutoGenerated,
		Threshold:        dm.Threshold,
		CalculationType:  string(dm.CalculationType),
		LatencyQuery:     string(dm.LatencyQuery),
		LatencyObjective: dm.LatencyObjective,
		ComponentID:      dm.ComponentID,
	}
	ml.Queries = make([]uint, len(dm.Queries))

	for i, q := range dm.Queries {
		ml.Queries[i] = q.ID
	}

	j, err := json.Marshal(ml)
	if err != nil {
		logrus.Errorf("Failed to encode json for Service.LogInfo: %s", err)
	}
	return &LogInfo{
		Type:  LogTypeMetric,
		ID:    ml.ID,
		Label: ml.Label,
		Data:  string(j),
	}
}

type queryLog struct {
	ID            uint   `json:"id"`
	Type          string `json:"type"`
	Query         string `json:"query"`
	AggregateType string `json:"aggregate_type"`
}

func (dq *Query) LogInfo() *LogInfo {
	ql := &queryLog{
		ID:            dq.ID,
		Type:          string(dq.QueryType),
		Query:         dq.Query,
		AggregateType: string(dq.AggregateType),
	}
	j, err := json.Marshal(ql)
	if err != nil {
		logrus.Errorf("Failed to encode json for Service.LogInfo: %s", err)
	}
	return &LogInfo{
		Type: LogTypeQuery,
		ID:   ql.ID,
		Data: string(j),
	}
}

type serviceDependencyLog struct {
	ID                  uint `json:"id"`
	RootServiceID       uint `json:"root_service_id"`
	DownstreamServiceID uint `json:"downstream_service_id"`
}

func (sd *ServiceDependency) LogInfo() *LogInfo {
	sdl := &serviceDependencyLog{

		ID:                  sd.ID,
		RootServiceID:       sd.RootServiceID,
		DownstreamServiceID: sd.DownstreamServiceID,
	}
	j, err := json.Marshal(sdl)
	if err != nil {
		logrus.Errorf("Failed to encode json for Service.LogInfo: %s", err)
	}
	return &LogInfo{
		Type: LogTypeServiceDependency,
		ID:   sdl.ID,
		Data: string(j),
	}
}

// We have two versions of SaveAPILog becase for updates (put) it
// would be messy to save the "before" api struct as these contain
// pointers to gorm fields that will change during the update. So for
// updates these we manually resolve LogInfo at the right time and
// then use SaveAPILogInfos on them. For everything else we can use
// the convenient SaveAPILoggables.

func SaveAPILogInfos(action string, before, after *LogInfo, ctx context.Context) {
	author := ""
	gu := goracleUser.GetUserFromContext(ctx)
	if gu != nil {
		author = gu.UID
	}
	lr := NewLogRecordFromLogInfos(action, before, after, author)
	err := GetCatalog().AddLogRecord(lr)
	if err != nil {
		logrus.Errorf("Failed to add logrecord for action %s: %s", action, err)
	}

}

func SaveAPILoggables(action string, before, after Loggable, ctx context.Context) {
	var bl, al *LogInfo
	if before != nil {
		bl = before.LogInfo()
	}
	if after != nil {
		al = after.LogInfo()
	}
	SaveAPILogInfos(action, bl, al, ctx)
}
