package catalog

import (
	"database/sql/driver"
	"fmt"

	"github.com/jinzhu/gorm"
)

// CalculationType defines the way that a metrics queries will be used to calculate availability
type CalculationType string

const (
	// ErrorRate uses a ErrorCount and a RequestCount query to determine availability
	ErrorRate CalculationType = "error_rate"
	// ThresholdAbove uses a Threshold Query. If that is above the threshold then is is available.
	ThresholdAbove = "threshold_above"
	// ThresholdBelow uses a Threshold Query. If that is below the threshold then is is available.
	ThresholdBelow = "threshold_below"
	// Latency uses Percentile Queries. If the selected latency query is below the threshold then it is available.
	Latency = "latency"
	// Mock is for test purposes only.
	Mock = "mock"
)

// Scan is necessary to retrieve a typed string CalculationType from the db
func (u *CalculationType) Scan(value interface{}) error {
	*u = CalculationType(value.([]byte))
	return nil
}

// Value is necessary to store a typed string CalculationType in the db
func (u CalculationType) Value() (driver.Value, error) { return string(u), nil }

// Metric Defines the schema for the metrics table and contains the information about a metric
type Metric struct {
	gorm.Model
	Label            string
	Name             string
	Description      string     `gorm:"size:2000"`
	Features         []*Feature `gorm:"many2many:feature_metric;"`
	FeatureRollup    bool       // This is only for feature Rollups
	ComponentRollup  bool       // This is only for component Rollups
	AutoGenerated    bool
	Queries          []*Query
	Threshold        float64
	ComponentID      uint `gorm:"index:idx_metric_component_id"`
	Component        *Component
	LatencyQuery     QueryType
	LatencyObjective float64 `sql:"default:99.9"`
	CalculationType  CalculationType
}

// URL returns the base URL for the resource
func (j *Metric) URL() string {
	return fmt.Sprintf("/metrics/%d", j.ID)
}
