package emailreport

import (
	"bytes"
	"html/template"
	"io/ioutil"

	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/config"
	"github.com/aymerick/douceur/inliner"
)

// Generator sets up information for an email report
type Generator interface {
	ID() string
	Name() string
	Description() string
	ReportTime() string
	Subject() string
	Recipients() []string
	Data() interface{}
}

// Output is all the information needed to render an email template
type Output struct {
	Body       string
	Recipients []string
	Subject    string
}

// Render uses a generator to generate the information needed to send
// an email report
func Render(g Generator) Output {
	output := Output{}
	output.Subject = g.Subject()
	output.Recipients = g.Recipients()
	data := g.Data()
	body, err := RenderBody("reports/availability-overall.html.tmpl", data)
	if err != nil {
		// FIXME: Do something useful
		logrus.Println("Error rendering email body:", err)
	}

	// Inline the css
	html, err := inliner.Inline(body)
	if err != nil {
		logrus.Println("Error inlining CSS!", err)
		output.Body = body
	} else {
		output.Body = html
	}

	return output
}

// Send sends the output of the given generator to the recipients of the report from goracle-noreply.
func Send(g Generator) {
	output := Render(g)
	SendMail("goracle-noreply@twitch.tv", output.Recipients, nil, output.Subject, output.Body)
}

// RenderBody renders the email for an EmailReporter
func RenderBody(bodyTemplatePath string, args interface{}) (string, error) {
	// Load page template
	pagePath := config.GetEmailTemplatePath("layouts/page.html.tmpl")
	pageData, err := ioutil.ReadFile(pagePath)
	if err != nil {
		logrus.Printf("Error reading template: %s\n", err)
		return "", err
	}

	pageTemplate := template.New("page")
	pageTemplate, err = pageTemplate.Parse(string(pageData))
	if err != nil {
		logrus.Printf("Error parsing page template: %s\n", err)
		return "", err
	}

	// Load body template
	bodyPath := config.GetEmailTemplatePath(bodyTemplatePath)
	bodyData, err := ioutil.ReadFile(bodyPath)
	if err != nil {
		logrus.Printf("Error reading template: %s\n", err)
		return "", err
	}

	bodyTemplate := pageTemplate.New("body")
	bodyTemplate, err = bodyTemplate.Parse(string(bodyData))
	if err != nil {
		logrus.Println("Body template parse error:", err)
		return "", err
	}

	_, err = pageTemplate.ParseGlob(config.GetEmailTemplatePath("css/*"))
	if err != nil {
		logrus.Printf("Error loading CSS templates: %s\n", err)
		return "", err
	}

	_, err = pageTemplate.ParseGlob(config.GetEmailTemplatePath("partials/*"))
	if err != nil {
		logrus.Printf("Error loading partials: %s\n", err)
		return "", err
	}

	var b bytes.Buffer

	err = pageTemplate.Execute(&b, args)
	if err != nil {
		logrus.Printf("Error executing email template: %s\n", err)
		return "", err
	}

	return b.String(), nil
}
