package guardianauth

import (
	"time"

	"github.com/jinzhu/gorm"
	uuid "github.com/satori/go.uuid"
	"github.com/sirupsen/logrus"
	gormigrate "gopkg.in/gormigrate.v1"
)

type dbNonceBackend struct {
	db *gorm.DB
}

// NonceRecord stores nonce informatioon in a GORM database
type nonceRecord struct {
	Nonce       string `gorm:"primary_key"`
	RedirectURI string
	Expiry      time.Time
}

// NewDBNonceBackend initializes a new in-memory nonce store
func newDBNonceBackend(db *gorm.DB) nonceBackend {
	logrus.Info("Initializing nonce backend")
	// Migrate to create nonce table if necessary
	m := gormigrate.New(db, gormigrate.DefaultOptions, []*gormigrate.Migration{
		{
			ID: "201706091152",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Running nonce migration")
				type nonceRecord struct {
					Nonce       string `gorm:"primary_key"`
					RedirectURI string
					Expiry      time.Time
				}
				return tx.AutoMigrate(&nonceRecord{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return tx.DropTable("nonce_records").Error
			},
		},
	})

	if err := m.Migrate(); err != nil {
		logrus.Fatalf("Could not migrate: %v", err)
	}

	return &dbNonceBackend{
		db: db,
	}

}

// storeNonceInfo inserts nonce info into the backend and returns the ID
// that it was stored under.
func (nb *dbNonceBackend) storeNonceInfo(ni nonceInfo) (string, error) {
	nonce := uuid.NewV4().String()

	nr := nonceRecord{
		Nonce:       nonce,
		RedirectURI: ni.RedirectURI,
		Expiry:      ni.Expiry,
	}

	res := nb.db.Create(&nr)

	if res.Error != nil {
		logrus.Warn("Unable to store nonce:", res.Error)
		return "", res.Error
	}

	return nonce, nil
}

// getAndClearNonceInfo retrieves and checks expiration of the nonce information.
func (nb *dbNonceBackend) getAndClearNonceInfo(nonce string) (*nonceInfo, error) {
	// This function leaks nonces over time, but it's the best solution to
	// being able to track the redirect URI associated with the nonce.

	tx := nb.db.Begin()
	nr := nonceRecord{}
	res := tx.Where(&nonceRecord{Nonce: nonce}).First(&nr)

	if res.Error != nil {
		return nil, res.Error
	}

	ni := nonceInfo{
		RedirectURI: nr.RedirectURI,
		Expiry:      nr.Expiry,
	}

	// Delete the record to avoid reuse
	res = tx.Delete(nr)
	if res.Error != nil {
		tx.Rollback()
		return nil, res.Error
	}
	tx.Commit()
	return &ni, nil
}
