package guardianauth

import (
    "bytes"
    "errors"
    "io"
    "io/ioutil"
    "net/http"
    "net/url"

    "github.com/sirupsen/logrus"
)

//This file is from malachai's guardian integration test with few modifications:
//https://git-aws.internal.justin.tv/sse/malachai-smoca-tests/blob/master/malachai-e2e-tests/guardian_client.go#L37

// GuardianClientConfig holds all necessary config to operate guardianclientconfig
type GuardianClientConfig struct {
    Username     string
    Password     string
    ClientID     string
}

func (cfg *GuardianClientConfig) getLoginBody() io.ReadCloser {
    values := url.Values{}
    values.Set("username", cfg.Username)
    values.Set("password", cfg.Password)
    return ioutil.NopCloser(bytes.NewBuffer([]byte(values.Encode())))
}

// GuardianClient handles getting an authorization token to talk to the guardian protected gql
type GuardianClient struct {
    Cfg         GuardianClientConfig
    AccessToken string
    Logger      *logrus.Logger
}

func (c *GuardianClient) extractFragmentValues(redirectURL *url.URL) (url.Values, error) {
  values, err := url.ParseQuery(redirectURL.Fragment)
  if err != nil {
    return nil, err
  }
  return values, nil
}

func (c *GuardianClient) generateAccessToken() (accessToken string, err error) {
    client := &http.Client{
        CheckRedirect: func(req *http.Request, via []*http.Request) error {
            return http.ErrUseLastResponse
        },
    }

    reqURL, err := url.Parse(baseURL)
    if err != nil {
        return
    }

    reqValues := url.Values{}
    reqValues.Set("client_id", c.Cfg.ClientID)
    reqValues.Set("response_type", "token")
    reqValues.Set("state", "my-state")

    reqURL.Path = "/oauth2/authorize"
    reqURL.RawQuery = reqValues.Encode()

    reqHeader := http.Header{}
    reqHeader.Add("Content-Type", "application/x-www-form-urlencoded")

    request := &http.Request{
        Method: "POST",
        URL:    reqURL,
        Header: reqHeader,
        Body:   c.Cfg.getLoginBody(),
    }
    res, err := client.Do(request)

    if err != nil {
        c.Logger.Error(err)
        return
    }

    if res.StatusCode != http.StatusFound {
        c.Logger.Debugf("error generating auth code: %s", res.Status)
        strBody, strBodyErr := ioutil.ReadAll(res.Body)
        if strBodyErr != nil {
            c.Logger.Error(strBodyErr)
        }
        c.Logger.Debug(string(strBody))

        err = errors.New("received status: " + res.Status)
        return
    }

    location := res.Header.Get("Location")
    if location == "" {
        err = errors.New("location header not returned")
        return
    }

    resLocation, err := url.Parse(location)
    if err != nil {
        return
    }
    resValues, err := c.extractFragmentValues(resLocation)
    if err != nil {
      return
    }

    accessToken = resValues.Get("access_token")
    if accessToken == "" {
        err = errors.New("no access_token sent in response")
        return
    }
    return
}

func (c *GuardianClient) refreshAccessToken() (err error) {
    c.AccessToken, err = c.generateAccessToken()
    return
}

// GetAccessToken is used to get the guardian access token
func (c *GuardianClient) GetAccessToken() (accessToken string, err error) {
    if c.AccessToken == "" {
        err = c.refreshAccessToken()
        if err != nil {
            return
        }
    }
    accessToken = c.AccessToken
    return
}
