// Package guardianauth provides utility functions to implement oauth2
// authentication with guardian. It provides a small, simple interface suitable
// for use with goji. Implementation is deliberately hidden, only exposing functions
// for initialization and registering callbacks and middleware.
//
// Note that we use asiimov's validation functions, but don't use any of the
// callback or nonce functions.
package guardianauth

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"code.justin.tv/systems/guardian/guardian"
	"code.justin.tv/availability/goracle/goracleUser"

	"github.com/jinzhu/gorm"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
	"golang.org/x/oauth2"
)

type nonceInfo struct {
	RedirectURI string
	Expiry      time.Time
}

type nonceBackend interface {
	storeNonceInfo(ni nonceInfo) (string, error)
	getAndClearNonceInfo(nonce string) (*nonceInfo, error)
}

type guardianBridge struct {
	oauthConfig  *oauth2.Config
	nonceBackend nonceBackend
}

const (
	baseURL       = "https://guardian.internal.justin.tv"
	authURL       = "https://guardian.internal.justin.tv/authorize"
	tokenURL      = "https://guardian.internal.justin.tv/oauth2/token"
	checkTokenURL = "https://guardian.internal.justin.tv/oauth2/check_token"
)

var ErrNilToken = errors.New("guardianauth: nil token passed to CheckToken")

type guardianContextKey string

var bridge guardianBridge

// InitGuardianLocal initializes guardian with the appropriate secrets and configuration
// This variant uses an in-memory nonce storage backend.
func InitGuardianLocal(clientID string, clientSecret string) {
	oauthConfig := oauth2.Config{
		ClientID:     clientID,
		ClientSecret: clientSecret,
		Endpoint: oauth2.Endpoint{
			AuthURL:  authURL,
			TokenURL: tokenURL,
		},
	}

	bridge = guardianBridge{
		oauthConfig:  &oauthConfig,
		nonceBackend: NewLocalNonceBackend(),
	}
}

// InitGuardianDB initializes guardian with the appropriate secrets and configuration
// if you are using gorm to store your backend.
func InitGuardianDB(clientID string, clientSecret string, dsn string) {
	// parse the DSN and open a Gorm connection
	s := strings.SplitN(dsn, "://", 2)
	if len(s) != 2 {
		logrus.Fatalf("Couldn't parse catalogDB DSN URL %s", dsn)
	}
	// Open a connection to the database for the nonce
	nonceDB, err := gorm.Open(s[0], s[1])
	if err != nil {
		logrus.Fatalf("Couldn't open Gorm: %s", err.Error())
	}

	oauthConfig := oauth2.Config{
		ClientID:     clientID,
		ClientSecret: clientSecret,
		Endpoint: oauth2.Endpoint{
			AuthURL:  authURL,
			TokenURL: tokenURL,
		},
	}

	bridge = guardianBridge{
		oauthConfig:  &oauthConfig,
		nonceBackend: newDBNonceBackend(nonceDB),
	}
}

func (gb *guardianBridge) CheckToken(token *oauth2.Token) (tc *guardian.TokenCheck, err error) {
	if token == nil {
		err = ErrNilToken
		return
	}

	client := gb.oauthConfig.Client(context.Background(), token)
	resp, err := client.Get(checkTokenURL)
	if err != nil {
		err = fmt.Errorf("asiimov: error checking token: %s", err.Error())
		return
	}
	if resp != nil {
		defer func() {
			closeErr := resp.Body.Close()
			if err == nil {
				err = closeErr
			}
		}()
	}

	switch resp.StatusCode {
	case http.StatusOK:
	case http.StatusNotFound:
		return
	default:
		err = fmt.Errorf("asiimov: error checking token: received %v status code from %v instead of %v", resp.StatusCode, checkTokenURL, http.StatusOK)
		return
	}

	tc = new(guardian.TokenCheck)
	err = json.NewDecoder(resp.Body).Decode(tc)
	if err != nil {
		err = fmt.Errorf("asiimov: error decoding token: %s", err.Error())
		return
	}
	return
}

// RegisterHandlers register handlers for Oauth on a goji mux under /oauth2
func RegisterHandlers(mux *goji.Mux) {
	oauth2Mux := goji.SubMux()
	oauth2Mux.Handle(pat.Get("/callback"), Oauth2CallbackHandler())

	mux.Handle(pat.New("/oauth2/*"), oauth2Mux)
}

func handleCallback(w http.ResponseWriter, r *http.Request) (*oauth2.Token, *nonceInfo, error) {
	nonce := r.FormValue("state")
	ni, err := bridge.nonceBackend.getAndClearNonceInfo(nonce)

	if err != nil {
		// Error getting the nonce
		return nil, nil, err
	}

	if ni == nil {
		// No valid nonce found
		return nil, nil, errors.New("State mismatch: nonce not found")
	}

	if time.Now().After(ni.Expiry) {
		return nil, nil, errors.New("Nonce expired!")
	}

	authCode := r.FormValue("code")

	var exchangeErr error
	token, exchangeErr := bridge.oauthConfig.Exchange(oauth2.NoContext, authCode)

	return token, ni, exchangeErr
}

// Oauth2CallbackHandler is an HTTP handler to handle an oauth2 response from guardian.
// Typically you should register it via RegisterHandlers.
func Oauth2CallbackHandler() (handler http.Handler) {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		token, ni, err := handleCallback(w, r)
		if err != nil {
			http.Error(w, err.Error(), http.StatusUnauthorized)
			return
		}
		tokenHandler(token, *ni, w, r)
	})
}

// tokenHandler is called when we get a successful response from Guardian -
// we create a cookie to store the token for future requests and redirect to
// the original requested URL.
func tokenHandler(token *oauth2.Token, ni nonceInfo, w http.ResponseWriter, r *http.Request) {
	tokenJSON, err := json.Marshal(token)
	tokenStr := base64.StdEncoding.EncodeToString(tokenJSON)
	if err != nil {
		http.Error(w, "Unable to parse token:"+err.Error(), http.StatusInternalServerError)
		return
	}

	// Encode the token and store it on the cookie
	cAuth := http.Cookie{
		Name:    "scAuthToken",
		Value:   string(tokenStr),
		Path:    "/",
		Expires: token.Expiry,
	}
	http.SetCookie(w, &cAuth)
    // Extract the Username and also create a cookie for it
	user, found := authenticate(r.Context(), token)
	if !found {
		http.Error(w, "unexpected error during authentication", http.StatusInternalServerError)
	}
	cUser := http.Cookie{
		Name:    "scAuthUser",
		Value:   string(user.UID),
		Path:    "/",
		Expires: token.Expiry,
	}
	http.SetCookie(w, &cUser)

	// Redirect to the original destination
	http.Redirect(w, r, ni.RedirectURI, http.StatusSeeOther)
	return
}

// Gets guardian token from cookie
func getTokenFromCookie(r *http.Request) (*oauth2.Token, error) {
	cookie, err := r.Cookie("scAuthToken")
	if err != nil {
		return nil, err
	}

	// decode the AuthToken string that was encoded by the caller
	decodedAccessToken, err := base64.StdEncoding.DecodeString(cookie.Value)
	if err != nil {
		return nil, err
	}
	t := oauth2.Token{
		AccessToken:string(decodedAccessToken),
	}
	return &t, nil
}

func guardianAuth(token *oauth2.Token) (*guardian.User, error) {
	tokenCheck, err := bridge.CheckToken(token)
	if err != nil {
		return nil, err
	}

	if tokenCheck == nil {
		return nil, nil
	}

	return tokenCheck.User, nil
}

// checks for auth
func authenticate(ctx context.Context, token *oauth2.Token) (user *guardian.User, ok bool) {
	var guardianErr error
	user, guardianErr = guardianAuth(token)
	if guardianErr != nil {
		logrus.Warnf("Error checking token with guardian: %s", guardianErr.Error())
	}

	if user == nil {
		return user, false
	}

	return user, true
}

// GuardianValidation is middleware that checks for authentication via oauth2
// In the case of a missing or an invalid token, it will redirect to guardian
// after creating a nonce that also store information about what page to return to
// after authorization.
func GuardianValidation(inner http.Handler) (outer http.Handler) {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		valid := true
		token, err := getTokenFromCookie(r)
		if err != nil {
			// No token or improperly formatted token
			valid = false
		}

		// Validate the token and get the user represented by it
		user, ok := authenticate(r.Context(), token)
		if !ok {
			valid = false
		}
		if !valid {
			nonce, err := generateNonce(r)
			if err != nil {
				// We failed to store a nonce.
				http.Error(w, "Failed to generate and store nonce", http.StatusInternalServerError)
				return
			}
			http.Redirect(w, r, authCodeURL(r, nonce), http.StatusSeeOther)
			return
		}

		// Everything is good, add the user to the context and continue
		ctx := context.WithValue(r.Context(), guardianContextKey("user"), user)
		goracleUserInstance := &goracleUser.GoracleUser{
			UID:            user.UID,
			CN:             user.CN,
			EmployeeNumber: uint32(user.UIDNumber),
		}
		ctx = context.WithValue(ctx, goracleUser.GoracleUserKey, goracleUserInstance)
		inner.ServeHTTP(w, r.WithContext(ctx))
	})
}

// GuardianUser is middleware that checks for authentication via oauth2
// Unlike GuardianValidation, it will continue if unauthenticated, just with
// an anonymous user.
func GuardianUser(inner http.Handler) (outer http.Handler) {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		valid := true
		token, err := getTokenFromCookie(r)
		logrus.Debug("Auth token:", token)
		if err != nil || token == nil {
			// No token or improperly formatted token
			valid = false
		}

		// next, get guardian user
		user, err := GuardianUserFromAuthToken(token)
		if err != nil {
			logrus.Warnf("authentication failed: %s", err.Error())
		}

		// finally, get goracle user (with employeeid, etc)
		ctx := r.Context()
		if valid {
			// Everything is good, add the user to the context
			ctx = context.WithValue(ctx, guardianContextKey("user"), user)
			goracleUserInstance, _ := GoracleUserFromGuardianUser(user)
			ctx = context.WithValue(ctx, goracleUser.GoracleUserKey, goracleUserInstance)
		}
		inner.ServeHTTP(w, r.WithContext(ctx))
	})
}

func GuardianUserFromAuthToken(token *oauth2.Token) (*guardian.User, error) {
	var user *guardian.User
	if token != nil {
		// Validate the token and get the user represented by it
		var ok bool
		user, ok = authenticate(nil, token)
		if ok {
			return user, nil
		}
	}
	return nil, errors.New("could not authenticate with token")
}

func GoracleUserFromGuardianUser(user *guardian.User) (*goracleUser.GoracleUser, error){
	// Everything is good, add the user to the context
	return &goracleUser.GoracleUser{
		UID:            user.UID,
		CN:             user.CN,
		EmployeeNumber: uint32(user.UIDNumber),
	}, nil
}

// GetUserFromContext gets a guardian user if it's been retrieved from guardian
func GetUserFromContext(ctx context.Context) *guardian.User {
	// Don't particularly care if this fails, it's expected to happen frequently
	u, _ := ctx.Value(guardianContextKey("user")).(*guardian.User)
	return u
}

func generateNonce(r *http.Request) (string, error) {
	ni := nonceInfo{
		RedirectURI: r.RequestURI,
		Expiry:      time.Now().UTC().Add(2 * time.Minute),
	}

	nonce, err := bridge.nonceBackend.storeNonceInfo(ni)
	return nonce, err
}

func authCodeURL(r *http.Request, nonce string) string {
	// Construct redirect URI based on incoming scheme and host
	scheme := "https"
	redirectURI := scheme + "://" + r.Host + "/oauth2/callback"

	// Hack for if we're doing local development, everything else should be
	// https
	if strings.HasPrefix(r.Host, "localhost") || strings.HasPrefix(r.Host, "servicecatalog-local") {
		scheme = "http"
		redirectURI = scheme + "://" + strings.Split(r.Host, ":")[0] + ":3000" + "/oauth2/callback"
	}

	var buf bytes.Buffer
	buf.WriteString(authURL)
	v := url.Values{
		"response_type": {"code"},
		"client_id":     {bridge.oauthConfig.ClientID},
		"redirect_uri":  {redirectURI},
		"state":         {nonce},
	}
	buf.WriteByte('?')
	buf.WriteString(v.Encode())
	return buf.String()
}
