package messaging

import (
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"strings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ses"
	"github.com/domodwyer/mailyak"
	"github.com/russross/blackfriday"

	"code.justin.tv/availability/goracle/config"
	"code.justin.tv/availability/goracle/goracleUser"
	"code.justin.tv/availability/goracle/emailreport"

	ldapLibrary "code.justin.tv/qe/twitchldap"
	"github.com/sirupsen/logrus"
)

var DoNotReply string = "status-internal-do-not-reply@justin.tv"

type EmailClient interface {
	SendMail([]string, []string, string, string, string) error
}

var emailClient EmailClient

type SESEmailClient struct{}

func (s *SESEmailClient) SendMail(to []string, replyto []string, subjectString string, bodyHTMLString string, bodyPlainString string) error {
	awsConfig := &aws.Config{Region: aws.String("us-west-2")}
	svc := ses.New(session.New(awsConfig))

	dest := ses.Destination{ToAddresses: aws.StringSlice(to)}

	body := ses.Body{Html: &ses.Content{Data: aws.String(bodyHTMLString)}}
	subject := ses.Content{Data: aws.String(subjectString)}
	mess := ses.Message{Body: &body, Subject: &subject}
	input := ses.SendEmailInput{Destination: &dest, Message: &mess, Source: &DoNotReply, ReplyToAddresses: aws.StringSlice(replyto)}
	_, err := svc.SendEmail(&input)

	return err
}

type SMTPEmailClient struct {
	MailEndpoint string
}

func (s *SMTPEmailClient) SendMail(to []string, replyto []string, subjectString string, bodyHTMLString string, bodyPlainString string) error {
	// Create a new email - specify the SMTP host and auth
	mail := mailyak.New(s.MailEndpoint, nil)

	mail.Bcc(to...)
	// only support a single reply-to e-mail
	if len(replyto) > 0 {
		mail.From(replyto[0])
	} else {
		mail.From(DoNotReply)
	}
	mail.WriteBccHeader(false)

	mail.Subject(subjectString)

	mail.HTML().Set(bodyHTMLString)
	mail.Plain().Set(bodyPlainString)

	if err := mail.Send(); err != nil {
		return err
	}

	return nil
}

func sendEmailHandler(w http.ResponseWriter, r *http.Request) {
	defer r.Body.Close()
	mess := MessageInput{}
	err := json.NewDecoder(r.Body).Decode(&mess)

	gu := goracleUser.GetUserFromContext(r.Context())
	if config.Config.EnableGuardian && gu == nil {
		w.Header().Set("Content-Type", "application/json; charset=UTF-8")
		w.WriteHeader(401)
		w.Write([]byte("User must be logged in to send email."))
		return
	} else if !config.Config.EnableGuardian || os.Getenv("ENVIRONMENT") == "development" {
		mess.To = []string{DoNotReply}
		mess.ReplyTo = []string{DoNotReply}
	}
	footerTemplate := `
<div>
	<p>
		This e-mail was sent by %s regarding the service %s (https://status.internal.justin.tv/services/%s)
	</p>
	<p>
		This message was generated and sent by the <a href="https://status.internal.justin.tv">Twitch Internal Status Page</a>
	</p>
	<p>
	  For questions about the Status Page please reach out to 
	  <a href="https://twitch.slack.com/messages/C1Y27UWG2/" 
	  	target="_blank">#availability-team
		</a> on slack
	</p>
</div>`
	footer := ""
	if len(mess.ReplyTo) > 0 {
		footer = fmt.Sprintf(footerTemplate, gu.UID, mess.ServiceName, mess.ServiceID)
		// Validate replyTo addresses
		var replyToErr error
		for _, r := range mess.ReplyTo {
			parts := strings.Split(r, "@")
			if len(parts) != 2 {
				replyToErr = fmt.Errorf("invalid email address '%s'", r)
				break
			} else if parts[1] != "twitch.tv" && parts[1] != "justin.tv" && parts[1] != "amazon.com" {
				replyToErr = fmt.Errorf("refusing to send mail with external replyTo address '%s'", r)
				break
			}
		}
		if replyToErr != nil {
			w.WriteHeader(400)
			w.Write([]byte(fmt.Sprintf("%v", replyToErr.Error())))
			return
		}
	} else {
		// Can't send an email with no replyto
		w.WriteHeader(400)
		w.Write([]byte(fmt.Sprintf("must specify at least one reply-to address")))
		return
	}

	bodyPlain := mess.Body
	bodyHTML := fmt.Sprintf("%s<br/><br/>%s", blackfriday.MarkdownCommon([]byte(mess.Body)), footer)

	err = emailClient.SendMail(mess.To, mess.ReplyTo, mess.Subject, bodyHTML, bodyPlain)

	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	if err != nil {
		w.WriteHeader(500)
		w.Write([]byte(fmt.Sprintf("%v", err.Error())))
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
}

func SendEmailNotification(toUID string, ccUIDs []string, initiatorName string, subject string, body string) {
	if !config.IsProduction() {
		logrus.Debugf("Sending slack message in non-prod to %s, limiting to team-members: %s", toUID, teamIDs)
		if !IsDevTeamMember(toUID) {
			logrus.Debugf("Do not send dev-test message to %s", toUID)
			return
		}
	}
	ldapClient, err := ldapLibrary.NewClient()
	if err != nil {
		logrus.Errorf("failed to initialize ldap client: %s", err.Error())
		return
	}
	toUser, err := ldapClient.GetUserInfoByName(toUID)
	if err != nil {
		logrus.Errorf("failed to get user info from ldap client for %s: %s", toUID, err.Error())
		return
	}
	ccEmails := make([]string, 0)
	for _, user := range ccUIDs {
		if !config.IsProduction() {
			logrus.Debugf("Sending slack message in non-prod to %s, limiting to team-members: %s", user, teamIDs)
			if !IsDevTeamMember(user) {
				logrus.Debugf("Do not send dev-test message to %s", user)
				continue
			}
		}
		ccUser, err := ldapClient.GetUserInfoByName(user)
		if err != nil {
			logrus.Errorf("failed to get user info from ldap client for %s: %s", user, err.Error())
			continue
		}
		ccEmails = append(ccEmails, ccUser.Mail)
	}

	logrus.Infof("sending email to %s", toUser.Mail)
	htmlBody := strings.Replace(body,"\n", "<br/>", -1)
	htmlBody += fmt.Sprintf("<br/><br/>Click <a href='%s/serviceOwner/%s'>here</a> to take action in the service catalog.",
		config.ServiceCatalogURL(), toUID)
	htmlBody += fmt.Sprintf("<br/><br/>If you have any questions or concerns, please reach out to %s. We are also here to help: #service-catalog", initiatorName)
	emailreport.SendMail("goracle-noreply@justin.tv", []string{toUser.Mail}, ccEmails, subject, htmlBody)
}
