package messaging

import (
	"encoding/json"
	"fmt"
	"net/http"
	"github.com/signalsciences/slashcmd/convert"
	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/config"
	"errors"
	"github.com/nlopes/slack"
	"strings"
	"sync"
	"time"
	"strconv"
	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/goracle/goracleUser"
)

var dummyChannel = "oe-tools-testing"

// simple HTML, to be used when user validates or self-assigns ownership
const SLACK_CONFIRMATION_HTML =`
	<p align='center' style='font-weight: bold'>
		</br>You have successfully %s the Primary Ownership for <a href='%s'>your service</a>.</br></br>Thank you!
	</p>`

// detailed HTML, to be used when user invalidates ownership
const SLACK_CONFIRMATION_DETAILED_HTML =`
	<p align='center' style='font-weight: bold'>
		</br>You have successfully invalidated the Primary Ownership for <a href='%s'>your service</a>. Please take the following actions:</br>
		</br>1. Find out who should be the owner</br>
		</br>2. Reach out to them for confirmation</br>
		</br>3. Go to <a href='%s'>My services</a> to transfer ownership of the service.  The new owner will get a slack message from service catalog informing them of ownership change and asking them to validate.</br>
		</br></br>
		</br>Thank you!
	</p>`

const SLACK_CHANNEL_MINUTES_EXPIRY = 10
const SLACK_USER_HOURS_EXPIRY = 12

const systemSlackMsgAttempted = "systemSlackMsgAttempted"
const systemActionableSlackMsgSuccess = "systemActionableSlackMsgSuccess"
const systemFYISlackMsgSuccess = "systemFYISlackMsgSuccess"
const systemSlackMsgFailure = "systemSlackMsgFailure"

const manualSlackMsgAttempted = "manualSlackMsgAttempted"
const manualSlackMsgSuccess = "manualSlackMsgSuccess"
const manualSlackMsgFailure = "manualSlackMsgFailure"

const slackMessageOwnershipValidated = "slackMessageOwnershipValidated"
const slackMessageOwnershipInvalidated = "slackMessageOwnershipInvalidated"
const slackMessageOwnershipValidationFailed = "slackMessageOwnershipValidationFailed"

var slackUserRequestMutex sync.RWMutex
var slackUserCacheLastUpdated time.Time
var emailToID map[string]slack.User
var nameToID map[string]slack.User

func GetSlackUser(userUID string, name string, client *slack.Client) (slack.User, error) {
	err := refreshSlackUserCacheIfOld(client)
	if err != nil {
		logrus.Error("error while refreshing slack users", err.Error())
		return slack.User{}, err
	}
	slackUserRequestMutex.RLock()
	defer slackUserRequestMutex.RUnlock()
	twitchEmail := userUID + "@twitch.tv"
	justinEmail := userUID + "@justin.tv"
	if user, ok := emailToID[twitchEmail]; ok {
		return user, nil
	}
	if user, ok := emailToID[justinEmail]; ok {
		return user, nil
	}
	if user, ok := nameToID[name]; ok {
		return user, nil
	}
	return slack.User{}, errors.New(fmt.Sprintf("Could not find uid in slack users %s", userUID))
}

func refreshSlackUserCacheIfOld(client *slack.Client) error {
	if emailToID != nil && time.Now().Sub(slackUserCacheLastUpdated).Hours() < SLACK_USER_HOURS_EXPIRY {
		return nil
	}

	slackUserRequestMutex.Lock()
	defer slackUserRequestMutex.Unlock()
	// Replicate check to make sure two requests don't block here
	if emailToID != nil && time.Now().Sub(slackUserCacheLastUpdated).Hours() < SLACK_USER_HOURS_EXPIRY {
		return nil
	}
	users, err := client.GetUsers()
	if err != nil {
		return err
	}
	logrus.Debug("Refreshing slack user cache")
	emailToID = make(map[string]slack.User)
	nameToID = make(map[string]slack.User)
	for _, user := range users {
		if user.Profile.Email == "" || user.ID == "" || user.Profile.RealName == "" {
			continue
		}
		emailToID[user.Profile.Email] = user
		nameToID[user.Profile.RealName] = user
	}
	slackUserCacheLastUpdated = time.Now()
	return nil
}

// isSlackChannelName determines whether the id is a slack shannel id or name. An example channel is https://twitch.slack.com/messages/C8C8GEHEU
func IsSlackChannelName(channelIDOrName string) bool {
	return !(len(channelIDOrName) == 9 && channelIDOrName[0] == 'C')
}

var slackChannelMutex sync.RWMutex
var slackChannelCacheLastUpdated time.Time
var channelIDToName map[string]slack.Channel
var channelNameToID map[string]string

func GetSlackChannelIDFromName(name string, client *slack.Client) (string, error) {
	if name == "" {
		return "", nil
	}
	err := refreshSlackChannelCacheIfOld(client)
	if err != nil {
		logrus.Error("Error while refreshing slack channels", err.Error())
		return "", err
	}
	chName := strings.TrimLeft(string(name), "#")
	slackChannelMutex.RLock()
	defer slackChannelMutex.RUnlock()
	id, ok := channelNameToID[chName]
	if !ok {
		logrus.Error("Could not find channel's ID. name is ", chName)
		return "", errors.New(fmt.Sprintf("Could not find the given slack channel id from name %s", chName))
	}
	return id, nil
}

func GetSlackChannelInfo(channelID string, client *slack.Client) (slack.Channel, error) {
	err := refreshSlackChannelCacheIfOld(client)
	if err != nil {
		logrus.Error("Error while refreshing slack channels", err.Error())
		return slack.Channel{}, err
	}
	slackChannelMutex.RLock()
	defer slackChannelMutex.RUnlock()
	channel, ok := channelIDToName[channelID]
	if !ok {
		return slack.Channel{}, errors.New(fmt.Sprintf("Could not find the given slack channel id %s", channelID))
	}
	return channel, nil
}

func refreshSlackChannelCacheIfOld(client *slack.Client) error {
	if channelIDToName != nil && time.Now().Sub(slackChannelCacheLastUpdated).Minutes() < SLACK_CHANNEL_MINUTES_EXPIRY {
		return nil
	}
	slackChannelMutex.Lock()
	defer slackChannelMutex.Unlock()
	if channelIDToName != nil && time.Now().Sub(slackChannelCacheLastUpdated).Minutes() < SLACK_CHANNEL_MINUTES_EXPIRY {
		return nil
	}
	channels, err := client.GetChannels(false)
	if err != nil {
		return err
	}
	logrus.Debug("Refreshing Slack Channel Cache")
	channelIDToName = make(map[string]slack.Channel)
	channelNameToID = make(map[string]string)
	for _, channel := range channels {
		channelIDToName[channel.ID] = channel
		channelNameToID[channel.Name] = channel.ID
	}
	slackChannelCacheLastUpdated = time.Now()
	return nil
}

func PostToGA(userID string, serviceId string, msg string) {
	gaEventFormat := "https://www.google-analytics.com/collect?v=1&tid=%s&cid=%s&t=event&ec=%s&ea=%s&el=%s"
	gaEvent := fmt.Sprintf(gaEventFormat, config.GetGATrackingID(), userID, "slack", msg, serviceId)
	resp, err := http.Get(gaEvent)
	if err != nil {
		logrus.Errorf("Failed to post to Google Analytics: userID=%s, msg=%s, serviceID=%s", userID, msg, serviceId)
		return
	}
	logrus.Debug(resp)
	defer resp.Body.Close()
}

// Post a Direct Message to the new Primary Owner
func SlackPostDM(userID, userName, subject, body, serviceName, serviceId string, includeButtons bool, customButtons []slack.Attachment) error {

	PostToGA(userID, serviceId, systemSlackMsgAttempted)

	validate := &slack.AttachmentAction{
		Name:  "Validate",
		Text:  "Validate",
		Type:  "button",
		Style: "primary",
		URL:   fmt.Sprintf("%s/messaging/serviceAudit/?serviceID=%s&action=validated&auditValue=%s", config.ServiceCatalogAPIURL(), serviceId, strings.Replace(userName, " ", "%20", -1)),
	}
	invalidate := &slack.AttachmentAction{
		Name:  "Invalidate",
		Text:  "Invalidate",
		Type:  "button",
		Style: "danger",
		URL:   fmt.Sprintf("%s/messaging/serviceAudit/?serviceID=%s&action=invalidated&auditValue=%s", config.ServiceCatalogAPIURL(), serviceId, strings.Replace(userName, " ", "%20", -1)),
	}
	var attachment []slack.Attachment
	if body != "" {
		md := convert.Github(body)
		attachment = []slack.Attachment{
			{
				Title:      fmt.Sprintf("Service: %s", serviceName),
				MarkdownIn: []string{"text"},
				Color:      "warning",
				Footer:     fmt.Sprintf("<%s|Twitch Service Catalog>", config.ServiceCatalogURL()),
				Fields: []slack.AttachmentField{
					{
						Title: subject,
						Value: md,
						Short: false,
					},
				},
			},
		}
		if includeButtons {
			var attachmentButtons []slack.Attachment
			if customButtons == nil { // use default validate/invalidate buttons
				attachmentButtons = []slack.Attachment{
					{
						Actions:    []slack.AttachmentAction{*validate, *invalidate},
					},
				}
			} else { // use custom buttons provided by caller
				attachmentButtons = customButtons
			}
			attachment = append(attachment, attachmentButtons...)
		}
	}

	slackToken := config.Config.SlackToken
	client := slack.New(slackToken)
	user, err := GetSlackUser(userID, userName, client)

	if err != nil {
		logrus.Errorf("Could not find Slack User for userID: %s, userName: %s. Error: %s ", userID, userName, err)
		PostToGA(userID, serviceId, systemSlackMsgFailure)
		return err
	}

	var channel string
	if !config.IsProduction() {
		logrus.Debugf("Sending slack message in non-prod to %s, limiting to team-members: %s", userID, teamIDs)
		if !IsDevTeamMember(userID) {
			logrus.Debugf("Do not send dev-test message to %s", userID)
			return nil
		}
	}

	_, _, channel, err = client.OpenIMChannel(user.ID)
	if err != nil {
		logrus.Errorf("Could not open Slack channel for userID: %s, userName: %s. Error: %s ", userID, userName, err)
		PostToGA(userID, serviceId, systemSlackMsgFailure)
		return err
	}

	msg := slack.PostMessageParameters{
		Channel:     channel,
		Attachments: attachment,
		Username:    "service-catalog-bot",
	}

	logrus.Debugf("Message from Service Catalog: channel - %s, msg - %s", channel, msg)
	channel, _, err = client.PostMessage(channel, "Message from Service Catalog", msg)
	if err != nil {
		logrus.Errorf("Could not post slack-message to channel: %s, msg: %s. Error: %s ", channel, msg, err)
		PostToGA(userID, serviceId, systemSlackMsgFailure)
		return fmt.Errorf("slack post failed: %s", err)
	}

	if includeButtons {
		PostToGA(userID, serviceId, systemActionableSlackMsgSuccess)
	} else {
		PostToGA(userID, serviceId, systemFYISlackMsgSuccess)
	}
	return nil

}

func slackPost(author, channel, subject, body, serviceName, serviceId string) error {
	PostToGA(author, serviceId, manualSlackMsgAttempted)
	var attach []slack.Attachment
	if body != "" {
		md := convert.Github(body)
		attach = []slack.Attachment{
			{
				AuthorName: author,
				Title:      fmt.Sprintf("Service: %s", serviceName),
				TitleLink:  fmt.Sprintf("%s/services/%s", config.ServiceCatalogURL(), serviceId),
				MarkdownIn: []string{"text"},
				Color:      "warning",
				Footer:     fmt.Sprintf("<%s|Twitch Service Catalog>", config.ServiceCatalogURL()),
				Fields: []slack.AttachmentField{
					{
						Title: subject,
						Value: md,
						Short: false,
					},
				},
			},
		}
	}

	msg := slack.PostMessageParameters{
		Channel:     channel,
		Attachments: attach,
		Username:    "service-catalog-bot",
	}
	slackToken := config.Config.SlackToken
	client := slack.New(slackToken)
	channel, _, err := client.PostMessage(channel, "Message from Service Catalog", msg)
	if err != nil {
		logrus.Errorf("Could not post slack-message to channel: %s, msg: %s. Error: %s ", channel, msg, err)
		PostToGA(author, serviceId, manualSlackMsgFailure)
		return fmt.Errorf("slack post failed: %s", err)
	}
	PostToGA(author, serviceId, manualSlackMsgSuccess)
	return nil
}

func sendSlackHandler(w http.ResponseWriter, r *http.Request) {
	defer r.Body.Close()

	mess := MessageInput{}
	err := json.NewDecoder(r.Body).Decode(&mess)

	gu := goracleUser.GetUserFromContext(r.Context())
	if config.Config.EnableGuardian && gu == nil {
		w.Header().Set("Content-Type", "application/json; charset=UTF-8")
		w.WriteHeader(401)
		w.Write([]byte("User must be logged in to post to Slack."))
		return
	} else if !config.IsProduction() {
		mess.To = []string{dummyChannel}
		logrus.Info("Sending Slack message in Dev, Redirecting to dummychannel ", dummyChannel)
	}

	author := "unknown"
	if gu != nil {
		author = gu.UID
	}

	for _, channel := range mess.To {
		err = slackPost(author, channel, mess.Subject, mess.Body, mess.ServiceName, mess.ServiceID)
		if err != nil {
			logrus.Errorf("slack post to %s failed: %s", channel, err)
			w.WriteHeader(500)
			w.Write([]byte(fmt.Sprintf("slack message post failed: %s", err)))
			return
		}
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
}

func slackValidationHandler(w http.ResponseWriter, r *http.Request) {
	var action = r.URL.Query()["action"][0]
	var serviceID = r.URL.Query()["serviceID"][0]
	var auditValue = r.URL.Query()["auditValue"][0]
	var serviceURL = r.URL.Host + "/services/" + serviceID
	var myServicesURL = r.URL.Host + "/serviceOwner/"
	var user string

	gu := goracleUser.GetUserFromContext(r.Context())
	if gu == nil {
		fmt.Fprintf(w, "Failed to get authenticated user")
		logrus.Errorf("Failed to get authenticated user")
		PostToGA("unknown", serviceID, slackMessageOwnershipValidationFailed)
		return
	}
	user = gu.UID
	id, err := strconv.ParseUint(serviceID, 10, 64)
	if err != nil {
		fmt.Fprintf(w, "Failed to parse serviceID")
		logrus.Errorf("Failed to parse serviceID: %s", serviceID)
		PostToGA(user, serviceID, slackMessageOwnershipValidationFailed)
		return
	}
	if action != "validated" && action != "invalidated" {
		fmt.Fprintf(w, "Invalid action: %s", action)
		logrus.Errorf("Invalid action: %s", action)
		PostToGA(user, serviceID, slackMessageOwnershipValidationFailed)
		return
	}

	// Create validation struct
	serviceAudit := &catalog.ServiceAudit{Auditor: user, ServiceID: uint(id), Action: action, AuditType: "owner", AuditValue: auditValue}
	serviceAudit.AuditTime = time.Now()

	// Add service audit
	err = catalog.GetCatalog().AddServiceAudit(serviceAudit)
	if err != nil {
		fmt.Fprintf(w, "Failed to add service audit")
		logrus.Errorf("Failed to add service audit for serviceID: %s", serviceID)
		PostToGA(user, serviceID, slackMessageOwnershipValidationFailed)
		return
	}

	if action == "validated" {
		// Display confirmation page with basic html
		fmt.Fprintf(w, SLACK_CONFIRMATION_HTML, action, serviceURL)
		// post to Google Analytics
		PostToGA(user, serviceID, slackMessageOwnershipValidated)
	} else {
		// Display confirmation page with detailed messaging for next steps
		fmt.Fprintf(w, SLACK_CONFIRMATION_DETAILED_HTML, serviceURL, myServicesURL)
		// post to Google Analytics
		PostToGA(user, serviceID, slackMessageOwnershipInvalidated)
	}
}

func slackAssignOwnerHandler(w http.ResponseWriter, r *http.Request) {
	// parse params
	var sid = r.URL.Query()["serviceID"][0]
	var eid = r.URL.Query()["ownerEmployeeId"][0]

	serviceId, err := strconv.ParseUint(sid, 10, 64)
	if err != nil {
		fmt.Fprintf(w, "Failed to parse serviceID: %s", err.Error())
		return
	}
	ownerEmpId, err := strconv.ParseUint(eid, 10, 64)
	if err != nil {
		fmt.Fprintf(w, "Failed to parse ownerEmployeeId: %s", err.Error())
		return
	}

	// get service
	service, err := catalog.GetCatalog().GetServiceByID(uint(serviceId))
	if err != nil {
		fmt.Fprintf(w, "Failed to retrieve service info: %s", err.Error())
		return
	}

	// override primery owner and persist in db (AddService() also updates)
	beforeLog := service.LogInfo()
	service.PrimaryOwnerID = uint(ownerEmpId)
	err = catalog.GetCatalog().AddService(service)
	if err != nil {
		fmt.Fprintf(w, "Failed to update service: %s", err.Error())
		return
	}
	afterLog := service.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, beforeLog, afterLog, r.Context())

	gu := goracleUser.GetUserFromContext(r.Context())
	if gu == nil {
		fmt.Fprintf(w, "Failed to get authenticated user")
		return
	}

	// Create validation struct
	serviceAudit := &catalog.ServiceAudit{Auditor: gu.UID, ServiceID: uint(serviceId), Action: "validated", AuditType: "owner", AuditValue: gu.CN}
	serviceAudit.AuditTime = time.Now()

	// Add service audit
	err = catalog.GetCatalog().AddServiceAudit(serviceAudit)
	if err != nil {
		fmt.Fprintf(w, "Failed to add service audit")
		return
	}

	// Display confirmation page
	var serviceURL = r.URL.Host + "/services/" + sid
	fmt.Fprintf(w, SLACK_CONFIRMATION_HTML, "self-assigned", serviceURL)
}