package twitchldap

import (
	"crypto/tls"
	"crypto/x509"
	"fmt"
	"strconv"
	"strings"

	"regexp"

	"time"

	"github.com/sirupsen/logrus"
	"gopkg.in/ldap.v2"
)

// EmployeeInfo is the main struct you want to use when getting LDAP info
type EmployeeInfo struct {
	EmployeeNumber uint32
	CN             string
	UID            string
	Manager        string
	ManagerLDAP    string
	Department     string
	Title          string
	Mail           string
	TwitchAmznUID  string
	EmployeeType   string
	LoginShell     string
	BuildingName   string
	Inactive       bool
	GivenName      string
	Surname        string
	PreferredName  string
	InactiveTime   time.Time
}

// NewLDAPClient returns an LDAP connection
// that is connected to the Twitch internal LDAP
// server, ready to issue searches
// The connection is anonymous and read only
// The client uses the singleton config to generate a sane
// client with proper options
const ldapEndpoint = "ldap-usw2.internal.justin.tv:636"
const dsSyncHistory = "ds-sync-hist"

// empFields are the fields we ask from LDAP. It should be nearly 1:1 with Employee Info struct
var empFields = []string{
	"cn",
	"uid",
	"employeeNumber",
	"givenName",
	"manager",
	"department",
	"title",
	"mail",
	"twAmazonUID",
	"employeeType",
	"loginShell",
	"sn",
	"buildingName",
	dsSyncHistory,
}

// LDAPClient is the main LDAPClient you should use
type LDAPClient struct {
	conn LDAPSearchable
}

// LDAPSearchable is a testing interface that is the same as the underlying ldap library
type LDAPSearchable interface {
	Search(*ldap.SearchRequest) (*ldap.SearchResult, error)
	Close()
}

// NewClient returns an LDAP connection
// that is connected to the Twitch internal LDAP
// server, ready to issue searches
// The connection is anonymous and read only
// The client uses the singleton config to generate a sane
// client with proper options
func NewClient() (*LDAPClient, error) {
	return NewClientWith(ldapEndpoint)
}

// NewClientWith returns an LDAP connection
// that is connected to the specified endpoint (host:port)
func NewClientWith(endpoint string) (*LDAPClient, error) {
	tlsConfig := &tls.Config{
		InsecureSkipVerify: true,
		ServerName:         strings.Split(endpoint, ":")[0],
		RootCAs:            x509.NewCertPool(),
	}

	c, err := ldap.DialTLS("tcp", endpoint, tlsConfig)
	if err != nil {
		return nil, err
	}
	return &LDAPClient{
		conn: c,
	}, nil
}

// WithConnection will allow overwriting the underlying connection (useful for tests)
func (c *LDAPClient) WithConnection(conn LDAPSearchable) *LDAPClient {
	c.conn = conn
	return c
}

func (c *LDAPClient) makeLDAPQuery(query string) (*ldap.SearchResult, error) {
	// Craft the search request
	searchRequest := ldap.NewSearchRequest(
		"ou=Users,dc=justin,dc=tv",
		ldap.ScopeWholeSubtree, ldap.NeverDerefAliases, 0, 0,
		false,
		query,
		empFields,
		nil,
	)
	// Do the search
	results, err := c.conn.Search(searchRequest)
	if err != nil {
		return nil, err
	}
	return results, nil
}

// GetUserInfo takes a user employeeNumber and looks up the user,
// returning the relevant userinfo object
// Returns nil,nil if the user is not found, and an nil,err if an error occurs
func (c *LDAPClient) GetUserInfo(employeeNumber uint32) (*EmployeeInfo, error) {
	results, err := c.makeLDAPQuery(fmt.Sprintf("(&(objectClass=person)(employeeNumber=%d))", employeeNumber))
	if err != nil {
		return nil, err
	}
	// if there are no entries in the result, the user wasn't found
	// so return nil
	if len(results.Entries) == 0 {
		return nil, nil
	} else if len(results.Entries) > 1 {
		// This shouldn't ever happen...
		return nil, fmt.Errorf("multiple users with employeeNumber '%d' found", employeeNumber)
	}
	// Extract the desired attribute values
	userInfo, err := extractUserInfo(results.Entries[0])
	// Check for errors, and also check if the returned employeeNumber matches what we asked for
	if err != nil {
		return nil, err
	} else if userInfo.EmployeeNumber != employeeNumber {
		return nil, fmt.Errorf("employee number in query result does not match with request")
	}
	return userInfo, nil
}

// GetUserInfoByName takes a ldap username and looks up the user,
// returning the relevant userinfo object
// Returns nil,nil if the user is not found, and an nil,err if an error occurs
func (c *LDAPClient) GetUserInfoByName(user string) (*EmployeeInfo, error) {
	results, err := c.makeLDAPQuery(fmt.Sprintf("(&(objectClass=person)(uid=%s))", user))
	if err != nil {
		return nil, err
	}

	if results == nil || len(results.Entries) == 0 {
		return nil, nil
	}
	// Extract the desired attribute values
	userInfo, err := extractUserInfo(results.Entries[0])
	// Check for errors, and also check if the returned user matches what we asked for
	if err != nil {
		return nil, err
	} else if userInfo.UID != user {
		return nil, fmt.Errorf("employee username in query result does not match with request")
	}
	return userInfo, nil
}

// GetAllUIDToUser returns a map of ldap usernames to user objects.
func (c *LDAPClient) GetAllUIDToUser() (map[string]EmployeeInfo, error) {
	allUsers, err := c.GetAllUserInfo()
	if err != nil {
		return nil, err
	}

	uidToUsers := make(map[string]EmployeeInfo)
	for _, u := range allUsers {
		uidToUsers[u.UID] = *u
	}

	return uidToUsers, nil
}

// GetAllManagerToReports returns a map containing all managers to reports
func (c *LDAPClient) GetAllManagerToReports() (map[string][]EmployeeInfo, error) {
	// Get UID to users to get uid
	uidToUser, err := c.GetAllUIDToUser()
	if err != nil {
		return nil, err
	}
	cnToUser := make(map[string]EmployeeInfo)

	for _, empInfo := range uidToUser {
		cnToUser[empInfo.CN] = empInfo
	}

	var managerToReports = make(map[string][]EmployeeInfo)
	for _, empInfo := range uidToUser {
		if empInfo.Manager == "" {
			fmt.Println("Encountered employee with no manager", empInfo)
			continue
		}
		managerCN := empInfo.Manager
		manager, ok := cnToUser[managerCN]
		if !ok {
			fmt.Println("Could not find manager in LDAP", empInfo, managerCN)
			continue
		}
		managerToReports[manager.UID] = append(managerToReports[manager.UID], empInfo)
	}
	return managerToReports, nil
}

// GetAllUserInfo returns information for every user in LDAP
// If any individual user causes an error, an error will be logged
// and the user will be omitted from the results
func (c *LDAPClient) GetAllUserInfo() ([]*EmployeeInfo, error) {
	// Craft the search request
	searchRequest := ldap.NewSearchRequest(
		"ou=Users,dc=justin,dc=tv",
		ldap.ScopeWholeSubtree, ldap.NeverDerefAliases, 0, 0,
		false,
		fmt.Sprintf("(&(objectClass=person)(employeeNumber=*))"),
		empFields,
		nil,
	)
	// Do the search
	results, err := c.conn.Search(searchRequest)
	if err != nil {
		return nil, err
	}
	// iterate over the resulting entries, transform them into EmployeeInfo's
	// and return the EmployeeInfo's
	userInfos := make([]*EmployeeInfo, 0)
	for _, entry := range results.Entries {
		ui, err := extractUserInfo(entry)
		if err != nil {
			fmt.Println(err.Error())
			logrus.Warnf("could not extract EmployeeInfo fields for LDAP DN '%s'", entry.DN)
			continue
		}
		userInfos = append(userInfos, ui)
	}
	return userInfos, nil
}

// Close closes the underlying connection of the LDAP Client
func (c *LDAPClient) Close() {
	c.conn.Close()
}

var managerRegex = regexp.MustCompile(`cn=([\w\- ().]*),`)

func extractCNFromDN(DN string) (string, error) {
	match := managerRegex.FindStringSubmatch(DN)
	if match == nil || len(match) < 2 {
		return "", fmt.Errorf("could not retrieve manager from `%s`", DN)
	}
	return match[1], nil
}

// looks for the cn and uid attributes and returns them
func extractUserInfo(data *ldap.Entry) (*EmployeeInfo, error) {
	var cn string
	var uid string
	var employeeNumber uint64 // will convert to uint32 later
	var err error
	var managerLD string
	var manager string
	var department string
	var title string
	var mail string
	var twAmazonUID string
	var employeeType string
	var loginShell string
	var buildingName string
	var inactive bool
	var inactivatedTime time.Time
	var surname string
	var givenName string
	eventTimes := make(map[string]string)
	for _, attr := range data.Attributes {
		if attr.Name == "cn" {
			// Make sure the cn information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple CN's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no CN found for DN '%s'", data.DN)
			}
			cn = attr.Values[0]
		} else if attr.Name == "uid" {
			// Make sure the uid information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple UID's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no UID found for DN '%s'", data.DN)
			}
			uid = attr.Values[0]
		} else if attr.Name == "employeeNumber" {
			// Make sure the employeeNumber information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple EmployeeNumber's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no EmployeeNumber found for DN '%s'", data.DN)
			}
			employeeNumber, err = strconv.ParseUint(attr.Values[0], 10, 32)
			if err != nil {
				return nil, fmt.Errorf("could not parse employeeNumber: %s", err.Error())
			}
			if employeeNumber == 0 {
				return nil, fmt.Errorf("employee number cannot be 0")
			}
		} else if attr.Name == "manager" {
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple managers found for DN '%s'", data.DN)
			}
			managerLD = attr.Values[0]
			manager, err = extractCNFromDN(managerLD)
			if err != nil {
				return nil, err
			}
		} else if attr.Name == "mail" {
			// Make sure the mail information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple mail's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no mail found for DN '%s'", data.DN)
			}
			mail = attr.Values[0]
		} else if attr.Name == "title" {
			// Make sure the title information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple title's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no title found for DN '%s'", data.DN)
			}
			title = attr.Values[0]
		} else if attr.Name == "twAmazonUID" {
			// Make sure the twAmazonUID information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple twAmazonUID's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no twAmazonUID found for DN '%s'", data.DN)
			}
			twAmazonUID = attr.Values[0]
		} else if attr.Name == "employeeType" {
			// Make sure the employeeType information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple employeeType's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no employeeType found for DN '%s'", data.DN)
			}
			employeeType = attr.Values[0]
		} else if attr.Name == "loginShell" {
			// Make sure the loginShell information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple loginShell's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no loginShell found for DN '%s'", data.DN)
			}
			loginShell = attr.Values[0]
		} else if attr.Name == "buildingName" {
			// Make sure the buildingName information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple buildingName's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 1 {
				buildingName = attr.Values[0]
			}
		} else if attr.Name == "department" {
			department = attr.Values[0]
		} else if attr.Name == dsSyncHistory {
			//references: https://backstage.forgerock.com/knowledge/kb/book/b93837928
			//https://docs.oracle.com/cd/E22289_01/html/821-1273/detecting-and-resolving-replication-inconsistencies.html

			// completely collect all the event ids and and corresponding timestamps
			for _, value := range attr.Values {
				tokens := strings.Split(value, ":")
				if strings.ToLower(tokens[0]) == "modifytimestamp" {
					csn := tokens[1]
					eventId := csn[0:20]
					eventTimes[eventId] = tokens[len(tokens)-1]
				}
			}

			// parse out the extended attribute values
			var disableEventID string
			for _, value := range attr.Values {
				tokens := strings.Split(value, ":") // attr name, eventid, operation, attr value
				switch tokens[0] {
				case "ds-pwp-account-disabled":
					if len(tokens) >= 4 {
						inactive = tokens[3] == "true"
						// Reference: https://backstage.forgerock.com/knowledge/kb/book/b74223337
						// Some attributes, such as ds-sync-state and ds-sync-hist give the CSN so you can determine the replication state. The CSN consists of the following information:
						// * The first 16 digits are a timestamp.
						// * The next 4 digits make up the replica-id (DS ID).
						csn := tokens[1]
						disableEventID = csn[0:20]
					} else {
						logrus.Printf("invalid ds-pwp-account-disabled: %s", value)
					}
					//README: handle other attributes in a new case in the future here if needed
				default:
					logrus.Debugf("no handler for extended attribute %s", tokens[0])
				}
			}

			// at this point, parse out timestamp
			if inactive {
				eventTime := eventTimes[disableEventID]
				if eventTime == "" {
					//logrus.Errorf("DEBUG: DN %s is inactive, but eventTime is empty", data.DN)
					//logrus.Printf("DEBUG: dumping all %s metadata:", dsSyncHistory)
					//for _, value := range attr.Values {
					//	logrus.Printf("DEBUG: %s entry: %s", dsSyncHistory, value)
					//}
					continue
				}
				inactivatedTime, err = time.Parse("20060102150405Z", eventTime)
				if err != nil {
					logrus.Errorf("unable to parse %s", eventTimes[disableEventID])
				}
			}

		} else if attr.Name == "sn" {
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple surnames found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no surname found for DN '%s'", data.DN)
			} else if len(attr.Values) == 1 {
				surname = attr.Values[0]
			}
		} else if attr.Name == "givenName" {
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple givenNames found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no givenName found for DN '%s'", data.DN)
			} else if len(attr.Values) == 1 {
				givenName = attr.Values[0]
			}
		}
	}
	// Verify that we found all the required fields before just blindly returning
	// This should never be triggered, because the response _should_ include
	// the attribute and just have len(Attribute.Values) == 0
	// Defensive programming!
	if cn == "" {
		return nil, fmt.Errorf("no CN found for DN '%s'", data.DN)
	}
	if uid == "" {
		return nil, fmt.Errorf("no UID found for DN '%s'", data.DN)
	}
	if employeeNumber == 0 {
		return nil, fmt.Errorf("no employeeNumber found for DN '%s'", data.DN)
	}

	ui := &EmployeeInfo{
		EmployeeNumber: uint32(employeeNumber),
		CN:             cn,
		UID:            uid,
		ManagerLDAP:    managerLD,
		Manager:        manager,
		Department:     department,
		Mail:           mail,
		TwitchAmznUID:  twAmazonUID,
		Title:          title,
		EmployeeType:   employeeType,
		LoginShell:     loginShell,
		BuildingName:   buildingName,
		Inactive:       inactive,
		InactiveTime:   inactivatedTime,
		GivenName:      givenName,
		Surname:        surname,
		PreferredName:  givenName + " " + surname,
	}
	return ui, nil
}
