package apiv2

import (
	"context"
	"encoding/json"
	"strings"

	graphql "github.com/neelance/graphql-go"
	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/catalog"
)

type attributeResolver struct {
	a *catalog.Attribute
}

func (r *Resolver) Attribute(args struct{ ID graphql.ID }) (*attributeResolver, error) {
	return resolveAttribute(args.ID)
}

func (r *Resolver) Attributes(
	args struct {
		ObjectType *string
		ObjectID   *graphql.ID
		Name       *string
		Value      *string
		Unique     *bool
	}) ([]*attributeResolver, error) {
	params := make(map[string]interface{})
	if args.ObjectType != nil {
		params["object_type"] = *args.ObjectType
	}
	if args.ObjectID != nil {
		id, err := idStringToUint(*args.ObjectID)
		if err != nil {
			logrus.Errorf("failed to convert object id %s to uint due to: %s", *args.ObjectID, err.Error())
			return nil, err
		}
		params["object_id"] = id
	}
	if args.Name != nil {
		params["name"] = *args.Name
	}
	if args.Value != nil {
		params["value"] = *args.Value
	}
	attrs, err := catalog.GetCatalog().GetAttributes(params)
	if err != nil {
		logrus.Errorf("failed to get attributes matching %#v due to: %s", params, err.Error())
		return nil, err
	}
	do_unique := false
	if args.Unique != nil {
		do_unique = *args.Unique
	}
	attributeResolvers := []*attributeResolver{}
	unique := make(map[string]map[string]bool)
	for _, attr := range attrs {
		if do_unique {
			_, ok := unique[attr.Name]
			if !ok {
				unique[attr.Name] = make(map[string]bool)
			}
			_, ok = unique[attr.Name][attr.Value]
			if !ok {
				unique[attr.Name][attr.Value] = true
				attributeResolvers = append(attributeResolvers, &attributeResolver{a: attr})
			}
		} else {
			attributeResolvers = append(attributeResolvers, &attributeResolver{a: attr})
		}
	}
	return attributeResolvers, nil
}

func resolveAttribute(id graphql.ID) (*attributeResolver, error) {
	u, err := idStringToUint(id)
	if err != nil {
		return nil, err
	}
	attr, err := catalog.GetCatalog().GetAttributeByID(u)
	if err != nil {
		return nil, err
	}
	return &attributeResolver{a: attr}, nil
}
func resolveAttributes(objectType string, objectId uint) ([]*attributeResolver, error) {
	params := make(map[string]interface{})
	params["object_type"] = objectType
	params["object_id"] = objectId

	attrs, err := catalog.GetCatalog().GetAttributes(params)
	if err != nil {
		return nil, err
	}
	attributeResolvers := []*attributeResolver{}
	for _, attr := range attrs {
		attributeResolvers = append(attributeResolvers, &attributeResolver{a: attr})
	}
	return attributeResolvers, nil
}

func (r *attributeResolver) ID() graphql.ID {
	return idUintToString(r.a.ID)
}

func (r *attributeResolver) ObjectType() string {
	return string(r.a.ObjectType)
}

func (r *attributeResolver) ObjectID() graphql.ID {
	return idUintToString(r.a.ObjectId)
}

func (r *attributeResolver) Name() string {
	return r.a.Name
}

func (r *attributeResolver) Value() string {
	return r.a.Value
}

// Mutations
type attributeInput struct {
	ID         *graphql.ID `json:"id"`
	Name       string      `json:"name"`
	Value      *string     `json:"value"`
	ObjectType string      `json:"object_type"`
	ObjectID   graphql.ID  `json:"object_id"`
	Deleted    *bool       `json:"-"` // This is only to allow deletion of associated attributes for deep saves
}

func (r *Resolver) saveAttributes(Attributes *[]*attributeInput, id graphql.ID, objectType string, ctx context.Context) {
	if Attributes != nil {
		for _, attr := range *Attributes {
			if attr.Deleted == nil || *attr.Deleted == false {
				attr.ObjectID = id
				attr.ObjectType = objectType
				if attr.ID == nil {
					var args struct{ Attribute *attributeInput }
					args.Attribute = attr
					r.CreateAttribute(ctx, args)
				} else {
					uint_id, _ := idStringToUint(*attr.ID)
					if uint_id == 0 {
						var args struct{ Attribute *attributeInput }
						args.Attribute = attr
						r.CreateAttribute(ctx, args)
					} else {
						var args struct {
							ID        graphql.ID
							Attribute *attributeInput
						}
						args.Attribute = attr
						args.ID = *attr.ID
						r.UpdateAttribute(ctx, args)
					}
				}
			} else {
				if attr.ID != nil {
					var args struct {
						ID graphql.ID
					}
					args.ID = *attr.ID
					r.DeleteAttribute(ctx, args)
				}
			}
		}
	}
}

func (ai attributeInput) LogInfo() *catalog.LogInfo {
	j, err := json.Marshal(ai)
	if err != nil {
		logrus.Errorf("Failed to encode json for Attribute.LogInfo: %s", err)
	}
	id, _ := idStringToUint(*ai.ID)
	return &catalog.LogInfo{
		Type:  catalog.LogTypeAttribute,
		ID:    id,
		Label: ai.Name + "-" + string(ai.ObjectID) + "-" + ai.ObjectType,
		Data:  string(j),
	}
}

func (r *Resolver) CreateAttribute(ctx context.Context,
	args struct {
		Attribute *attributeInput
	},
) (*attributeResolver, error) {

	// Create struct
	attr := &catalog.Attribute{}

	if err := mergeAttributeData(attr, args.Attribute); err != nil {
		logrus.Errorf("failed to get merge attribute data %#v and %#v due to: %s", attr, args.Attribute, err.Error())
		return nil, err
	}
	resolver, err := saveAttribute(attr)
	if err != nil {
		logrus.Errorf("failed to get save new attribute %#v due to: %s", attr, err.Error())
		return nil, err
	}
	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, attr, ctx)
	return resolver, nil
}

func (r *Resolver) UpdateAttribute(ctx context.Context,
	args struct {
		ID        graphql.ID
		Attribute *attributeInput
	},
) (*attributeResolver, error) {

	// Get the existing component
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		logrus.Errorf("failed to convert attribute id %s to uint due to: %s", args.ID, err.Error())
		return nil, err
	}
	attr, err := catalog.GetCatalog().GetAttributeByID(id)
	if err != nil {
		logrus.Errorf("failed to get attribute with ID %d due to: %s", id, err.Error())
		return nil, err
	}
	beforeLog := attr.LogInfo()
	// Update the given fields
	if err := mergeAttributeData(attr, args.Attribute); err != nil {
		logrus.Errorf("failed to get merge attribute data %#v and %#v due to: %s", attr, args.Attribute, err.Error())
		return nil, err
	}

	resolver, err := saveAttribute(attr)
	if err != nil {
		logrus.Errorf("failed to get update existing attribute %#v due to: %s", attr, err.Error())
		return nil, err
	}
	afterLog := attr.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, beforeLog, afterLog, ctx)
	// Save and give back a GraphQL resolver
	return resolver, nil
}

func (r *Resolver) DeleteAttribute(ctx context.Context,
	args struct {
		ID graphql.ID
	},
) (*attributeResolver, error) {

	// Get the existing component
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		logrus.Errorf("failed to convert attribute id %s to uint due to: %s", args.ID, err.Error())
		return nil, err
	}
	attr, err := catalog.GetCatalog().GetAttributeByID(id)
	if err != nil {
		logrus.Errorf("failed to get attribute with ID %d due to: %s", id, err.Error())
		return nil, err
	}
	// Delete it
	err = catalog.GetCatalog().DeleteAttribute(attr)
	if err != nil {
		logrus.Errorf("failed to deletee attribute with ID %s due to: %s", args.ID, err.Error())
		return nil, err
	}
	catalog.SaveAPILoggables(catalog.LogOpDelete, attr, nil, ctx)
	return &attributeResolver{a: attr}, nil
}

func saveAttribute(attribute *catalog.Attribute) (*attributeResolver, error) {
	err := catalog.GetCatalog().AddAttribute(attribute)
	if err != nil {
		return nil, err
	}
	return &attributeResolver{a: attribute}, nil
}

func mergeAttributeData(dbAttr *catalog.Attribute, apiAttr *attributeInput) error {
	dbAttr.Name = strings.ToLower(apiAttr.Name)
	dbAttr.ObjectType = apiAttr.ObjectType

	id, err := idStringToUint(apiAttr.ObjectID)
	if err != nil {
		return err
	}
	dbAttr.ObjectId = id
	// optional, merge with default
	if apiAttr.Value != nil {
		dbAttr.Value = *apiAttr.Value
	}
	return nil
}
