package apiv2

import (
	"context"
	"time"

	"code.justin.tv/availability/goracle/catalog"
	graphql "github.com/neelance/graphql-go"
	"code.justin.tv/availability/goracle/goracleUser"
	"github.com/sirupsen/logrus"
)

type serviceAuditResolver struct {
	sa *catalog.ServiceAudit
}

func (r *Resolver) ServiceAudits(
	args struct {
		ServiceID *graphql.ID
		Latest    *bool
	},
) ([]*serviceAuditResolver, error) {
	var serviceAudits []*catalog.ServiceAudit
	lookupMap := make(map[string]interface{})
	if args.ServiceID != nil {
		id, err := idStringToUint(*args.ServiceID)
		if err != nil {
			return nil, err
		}
		lookupMap["service_id"] = id
	}
	if args.Latest != nil && *args.Latest == false {
		serviceAudits = catalog.GetCatalog().GetServiceAudits(lookupMap)
	} else {
		serviceAudits = catalog.GetCatalog().GetLatestServiceAudits(lookupMap)
	}

	serviceAuditResolvers := []*serviceAuditResolver{}
	for _, sa := range serviceAudits {
		serviceAuditResolvers = append(serviceAuditResolvers, &serviceAuditResolver{sa: sa})
	}
	return serviceAuditResolvers, nil
}

func resolveServiceAudits(serviceID uint, latest bool) ([]*serviceAuditResolver, error) {
	var serviceAudits []*catalog.ServiceAudit
	lookupMap := make(map[string]interface{})
	lookupMap["service_id"] = serviceID
	if latest == false {
		serviceAudits = catalog.GetCatalog().GetServiceAudits(lookupMap)
	} else {
		serviceAudits = catalog.GetCatalog().GetLatestServiceAudits(lookupMap)
	}

	serviceAuditResolvers := []*serviceAuditResolver{}
	for _, sa := range serviceAudits {
		serviceAuditResolvers = append(serviceAuditResolvers, &serviceAuditResolver{sa: sa})
	}
	return serviceAuditResolvers, nil
}

func (r *serviceAuditResolver) ID() graphql.ID {
	return idUintToString(r.sa.ID)
}

func (r *serviceAuditResolver) ServiceID() graphql.ID {
	return idUintToString(r.sa.ServiceID)
}

func (r *serviceAuditResolver) AuditType() string {
	return r.sa.AuditType
}

func (r *serviceAuditResolver) Auditor() string {
	return r.sa.Auditor
}

func (r *serviceAuditResolver) AuditValue() string {
	return r.sa.AuditValue
}

func (r *serviceAuditResolver) Action() string {
	return r.sa.Action
}

func (r *serviceAuditResolver) AuditTime() string {
	return r.sa.AuditTime.Format("2006-01-02T15:04:05.999999999-07:00")
}

// Mutations
type serviceAuditInput struct {
	ServiceID graphql.ID `json:"id"`
	AuditType string     `json:"name"`
	Action    string     `json:"value"`
	AuditValue string    `json:"audit_value"`
}

func (r *Resolver) CreateServiceAudit(ctx context.Context,
	args struct {
		ServiceAudit *serviceAuditInput
	},
) (*serviceAuditResolver, error) {
	var user string
	gu := goracleUser.GetUserFromContext(ctx)
	if gu != nil {
		user = gu.UID
	} else {
		user = "anonymous"
	}
	id, err := idStringToUint(args.ServiceAudit.ServiceID)
	_, err = catalog.GetCatalog().GetServiceByID(uint(id))
	if err != nil {
		logrus.Errorf("failed to query service by ID %s due to: %s", id, err.Error())
		return nil, err
	}
	// Create struct
	serviceAudit := &catalog.ServiceAudit{Auditor: user, ServiceID: id, Action: args.ServiceAudit.Action, AuditType: args.ServiceAudit.AuditType, AuditValue: args.ServiceAudit.AuditValue}
	serviceAudit.AuditTime = time.Now()

	resolver, err := saveServiceAudit(serviceAudit)
	if err != nil {
		logrus.Errorf("failed to create audit log entry due to: %s", err.Error())
		return nil, err
	}
	return resolver, nil
}

func saveServiceAudit(serviceAudit *catalog.ServiceAudit) (*serviceAuditResolver, error) {
	err := catalog.GetCatalog().AddServiceAudit(serviceAudit)
	if err != nil {
		return nil, err
	}
	return &serviceAuditResolver{sa: serviceAudit}, nil
}
