package apiv2

import (
	"context"

	graphql "github.com/neelance/graphql-go"
	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/catalog"
)

// Queries
type componentResolver struct {
	c *catalog.Component
}

func (r *Resolver) Component(args struct{ ID graphql.ID }) (*componentResolver, error) {
	return resolveComponent(args.ID)
}

func (r *Resolver) Components(
	args struct {
		Label          *string
		ServiceID      *graphql.ID
		AttributeName  *string
		AttributeValue *string
	},
) ([]*componentResolver, error) {
	ids, err := idsWithAttribute(catalog.LogTypeComponent, args.AttributeName, args.AttributeValue)
	if err != nil {
		logrus.Warnln(err)
		return nil, err
	}
	params := make(map[string]interface{})
	// Add relevant filters
	if args.Label != nil {
		params["label"] = *args.Label
	}
	if args.ServiceID != nil {
		id, err := idStringToUint(*args.ServiceID)
		if err != nil {
			return nil, err
		}
		params["service_id"] = id
	}
	comps, err := catalog.GetCatalog().GetComponentsComplete(ids, params)
	if err != nil {
		return nil, err
	}
	compResolvers := []*componentResolver{}
	for _, comp := range comps {
		compResolvers = append(compResolvers, &componentResolver{c: comp})
	}
	return compResolvers, nil
}

func resolveComponent(id graphql.ID) (*componentResolver, error) {
	u, err := idStringToUint(id)
	if err != nil {
		return nil, err
	}
	if u == 0 {
		return nil, nil
	}
	component, err := catalog.GetCatalog().GetComponentByID(u)
	if err != nil {
		return nil, err
	}
	return &componentResolver{c: component}, nil
}

func (r *componentResolver) ID() graphql.ID {
	return idUintToString(r.c.ID)
}

func (r *componentResolver) Name() string {
	return r.c.Name
}

func (r *componentResolver) Label() string {
	return r.c.Label
}

func (r *componentResolver) Description() string {
	return r.c.Description
}

func (r *componentResolver) Rollup() bool {
	return r.c.Rollup
}

func (r *componentResolver) Service() (*serviceResolver, error) {
	if r.c.ServiceID != 0 {
		id := idUintToString(r.c.ServiceID)
		return resolveService(id)
	}
	return nil, nil
}

func (r *componentResolver) ServiceID() *graphql.ID {
	if r.c.ServiceID != 0 {
		id := idUintToString(r.c.ServiceID)
		return &id
	}
	return nil
}

func (r *componentResolver) Claimed() bool {
	return r.c.Claimed
}

func (r *componentResolver) Type() string {
	return r.c.Type
}

func (r *componentResolver) Metrics() []*metricResolver {
	metricResolvers := []*metricResolver{}
	for _, metric := range r.c.Metrics {
		metricResolvers = append(metricResolvers, &metricResolver{m: metric})
	}
	return metricResolvers
}

func (r *componentResolver) MetricIDs() []graphql.ID {
	metricIDs := []graphql.ID{}
	for _, metric := range r.c.Metrics {
		metricIDs = append(metricIDs, idUintToString(metric.ID))
	}
	return metricIDs
}

func (r *componentResolver) Attributes() []*attributeResolver {
	res, err := resolveAttributes(catalog.LogTypeComponent, r.c.ID)
	if err != nil {
		return nil
	}
	return res
}

// Mutations
type componentInput struct {
	ID          *graphql.ID
	Name        *string
	Label       *string
	Description *string
	Rollup      *bool
	Claimed     *bool
	Type        *string
	ServiceID   *graphql.ID
	MetricIDs   *[]graphql.ID
	Metrics     *[]metricInput
	Attributes  *[]*attributeInput `json:"-"`
}

type CreateComponentArgs struct {
	Component *componentInput
}

func (r *Resolver) CreateComponent(ctx context.Context,
	args CreateComponentArgs,
) (*componentResolver, error) {

	// Create struct
	component := catalog.DefaultComponent()

	if err := mergeComponentData(component, args.Component, ctx, r); err != nil {
		return nil, err
	}
	resolver, err := saveComponent(component)
	if err != nil {
		return nil, err
	}
	r.saveAttributes(args.Component.Attributes, resolver.ID(), catalog.LogTypeComponent, ctx)
	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, component, ctx)

	return resolver, nil
}

type UpdateComponentArgs struct {
	ID        graphql.ID
	Component *componentInput
}

func (r *Resolver) UpdateComponent(ctx context.Context,
	args UpdateComponentArgs,
) (*componentResolver, error) {
	// Get the existing component
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		return nil, err
	}
	component, err := catalog.GetCatalog().GetComponentByID(id)
	if err != nil {
		return nil, err
	}

	beforeLog := component.LogInfo()

	// Update the given fields
	if err := mergeComponentData(component, args.Component, ctx, r); err != nil {
		return nil, err
	}

	resolver, err := saveComponent(component)
	if err != nil {
		return nil, err
	}
	r.saveAttributes(args.Component.Attributes, resolver.ID(), catalog.LogTypeComponent, ctx)
	afterLog := component.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, beforeLog, afterLog, ctx)

	return resolver, nil

}

type DeleteComponentArgs struct {
	ID graphql.ID
}

func (r *Resolver) DeleteComponent(ctx context.Context,
	args DeleteComponentArgs,
) (*componentResolver, error) {

	// Get the existing component
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		return nil, err
	}
	component, err := catalog.GetCatalog().GetComponentByID(id)
	if err != nil {
		return nil, err
	}

	// Delete it
	err = catalog.GetCatalog().DeleteComponent(component)
	if err != nil {
		return nil, err
	}

	catalog.SaveAPILoggables(catalog.LogOpDelete, component, nil, ctx)

	return &componentResolver{c: component}, nil
}

func saveComponent(component *catalog.Component) (*componentResolver, error) {
	err := catalog.GetCatalog().AddComponent(component)
	if err != nil {
		return nil, err
	}
	return &componentResolver{c: component}, nil
}

func mergeComponentData(dbComp *catalog.Component, apiComp *componentInput, ctx context.Context, r *Resolver) error {
	if apiComp.Name != nil {
		dbComp.Name = *apiComp.Name
	}
	// optional, merge with default
	if apiComp.Label != nil {
		dbComp.Label = *apiComp.Label
	}
	if apiComp.Description != nil {
		dbComp.Description = *apiComp.Description
	}
	if apiComp.Rollup != nil {
		dbComp.Rollup = *apiComp.Rollup
	}
	if apiComp.ServiceID != nil {
		serviceRes, err := resolveService(*apiComp.ServiceID)
		if err != nil {
			return err
		}
		if serviceRes != nil {
			dbComp.ServiceID = serviceRes.s.ID
			dbComp.Service = serviceRes.s
		} else {
			dbComp.ServiceID = 0
			dbComp.Service = nil
		}
	}
	if apiComp.Type != nil {
		dbComp.Type = *apiComp.Type
	}
	if apiComp.Metrics != nil {
		metrics := []*catalog.Metric{}
		for _, metric := range *apiComp.Metrics {
			if metric.ID == nil {
				args := CreateMetricArgs{Metric: &metric}
				res, err := r.CreateMetric(ctx, args)
				if err != nil {
					return err
				}
				metrics = append(metrics, res.m)
			} else {
				id, err := idStringToUint(*metric.ID)
				if err != nil {
					return err
				}
				if id == 0 {
					args := CreateMetricArgs{Metric: &metric}
					res, err := r.CreateMetric(ctx, args)
					if err != nil {
						return err
					}
					metrics = append(metrics, res.m)
				} else {
					args := UpdateMetricArgs{ID: *metric.ID, Metric: &metric}
					res, err := r.UpdateMetric(ctx, args)
					if err != nil {
						return err
					}
					metrics = append(metrics, res.m)
				}
			}
		}
		dbComp.Metrics = metrics
	} else if apiComp.MetricIDs != nil {
		dbComp.Metrics = make([]*catalog.Metric, 0)
		for _, mid := range *apiComp.MetricIDs {
			met, err := resolveMetric(mid)
			if err != nil {
				return err
			}
			dbComp.Metrics = append(dbComp.Metrics, met.m)
		}
	}
	return nil
}

func (r *componentResolver) Logs(
	args struct {
		Offset *int32
		Limit  *int32
	}) []*logRecordResolver {

	offset := -1
	limit := -1

	if args.Offset != nil {
		offset = int(*args.Offset)
	}
	if args.Limit != nil {
		limit = int(*args.Limit)
	}

	res, err := resolveLogs("component", r.c.ID, offset, limit)
	if err != nil {
		return nil
	}
	return res
}
