package apiv2

import (
	contacts "code.justin.tv/amzn/MyFultonContactsServiceLambdaTwirp"
	lambdaTransport "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	"code.justin.tv/foundation/twitchclient"
	"context"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/neelance/graphql-go"
	"github.com/sirupsen/logrus"
	"net/http"
	"sync"
	"time"
)

// The pre-existing programming style for graphql implemntation seems to follow this example:
// https://gist.github.com/larryaasen/4fc744796e193da0aeeff671d1d3d3d7

// 1. Create a go file that maps to a resolver type.
// the name of this file should be in lowercase of the type in schema.go
// eg, Contact resource type in schema.go maps to contact.go

// 2. Create a resolver struct that contains at the minimum:
// i) underlying data-structure from the source data-store/API,
// ii) error field
// The resolver should be named by convention: <type>Resolver, eg contactResolver
type contactResolver struct {
	person *contacts.ProtoPerson
	err    error
}

// getContactsServiceClient is a helper function that returns an instance of the contacts service client
func getContactsServiceClient() (contacts.MyFultonContactsServiceService, error) {
	contactsConf := twitchclient.ClientConf{
		Host: "arn:aws:lambda:us-west-2:117703951204:function:MyFultonContactsServiceLambda-LambdaFunction-1FXZU65CZ6XXT",
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 75,
			TLSHandshakeTimeout: 15 * time.Second,
		},
		CheckRedirect: func(req *http.Request, via []*http.Request) error {
			return http.ErrUseLastResponse
		},
	}
	contactsTwirpClient := twitchclient.NewHTTPClient(contactsConf)
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	if err != nil {
		return nil, err
	}

	contactsTwirpConfig := sess.Config.WithHTTPClient(contactsTwirpClient).WithMaxRetries(3)
	contactsTwirpLambdaClient := lambda.New(session.Must(session.NewSession(contactsTwirpConfig)))

	contactsService := contacts.NewMyFultonContactsServiceServiceJSONClient("",
		lambdaTransport.NewClient(contactsTwirpLambdaClient, contactsConf.Host))

	return contactsService, nil
}

// 3.A For each api declared in schema.go (eg contact(req) response), implement a method with a
// camel-cased name (eg Contact) for the type (contact).
// NOTE that params must be wrapped inside a struct for graphql module to successfully validate the schema
func (r *Resolver) Contact(args struct{ UserId graphql.ID }) (*contactResolver, error) {
	// no-need to validate params, as the schema describes the type, and graphql module has built-in validation

	userReq := &contacts.UserRequest{UserId: string(args.UserId)}
	ctx := context.Background()
	contactsService, err := getContactsServiceClient()
	if err != nil {
		logrus.Fatalf("unable to get contacts service client due to: %s", err.Error())
		return nil, err
	}
	person, err := contactsService.GetPerson(ctx, userReq)
	if err != nil {
		return &contactResolver{person: nil, err: err}, err
	}
	manager, err := contactsService.GetManager(ctx, userReq)
	if err != nil {
		return &contactResolver{person: nil, err: err}, err
	}
	person.Person.Manager = manager.Person
	structInfo, err := contactsService.GetStructure(ctx, userReq)
	if err != nil {
		return &contactResolver{person: nil, err: err}, err
	}
	person.Person.Team = structInfo.Team
	person.Person.Team.Org = structInfo.Org
	person.Person.Team.Org.Bu = structInfo.Bu
	return &contactResolver{person: person.Person}, nil
}

// cache contacts service data for ListPersons for this long
const personsCacheTimeout = time.Duration(12 * time.Hour)

// contactsServiceCache holds the cached data, lastUpdated timestamp and a mutex lock
type contactsServiceCache struct {
	personsCache                *contacts.PersonsResponse
	personsCacheLastUpdated     time.Time
	sync                        sync.Mutex
}
var contactsServiceCacheData = contactsServiceCache{}

// update cache containing contacts service data for ListPersons if cache is stale
func (c *contactsServiceCache) updatePersonsCacheIfStale() error {
	// check if cache needs to be updated
	if time.Since(c.personsCacheLastUpdated) < personsCacheTimeout {
		logrus.Debugf("contacts cache fresh; not updating")
		return nil
	}
	logrus.Debugf("contacts cache stale; updating")
	ctx := context.Background()
	contactsService, err := getContactsServiceClient()
	// keep old cache if data refresh fails
	if err != nil {
		logrus.Warnf("keeping old contacts cache; unable to get contacts service client due to: %s", err.Error())
		return err
	}
	personsCacheUpdated, err := contactsService.ListPersons(ctx, &contacts.PersonsRequest{})
	if err != nil {
		logrus.Warnf("keeping old contacts cache; unable to list persons from contacts service due to: %s", err.Error())
		return err
	}
	c.sync.Lock()
	defer c.sync.Unlock()
	c.personsCacheLastUpdated = time.Now()
	c.personsCache = personsCacheUpdated
	return nil
}

// 3.B, for APIs that have slice of values as input parameters, follow this convention:
// i) in schema.go: apiName(param_name: [type]!): [response_type)]!, eg contact(user_id: [ID]!): [Contact]!
// ii) in resolver, the param declaration needs to be: args struct{ param_name []*type }, eg: args struct{ UserIds []*graphql.ID }
// Pay attention to the schema hint character: ! - non-optional), and declared type specification ([]*type - a slice of pointers of certain type)
// any other combination does not work.
func (r *Resolver) Contacts(args struct{ UserIds []*graphql.ID }) ([]*contactResolver, error) {
	resolvers := make([]*contactResolver, 0)
	err := contactsServiceCacheData.updatePersonsCacheIfStale()
	if err != nil {
		resolvers = append(resolvers, &contactResolver{person: nil, err: err})
		return resolvers, err
	}
	contactsServiceCacheData.sync.Lock()
	defer contactsServiceCacheData.sync.Unlock()
	for _, person := range contactsServiceCacheData.personsCache.Persons {
		foundInWhitelist := false
		for _, whitelisted := range args.UserIds {
			if string(*whitelisted) == person.UserId {
				foundInWhitelist = true
				break
			}
		}
		if !foundInWhitelist {
			continue
		}

		resolvers = append(resolvers, &contactResolver{person: person, err: nil})
	}
	return resolvers, nil
}

func (r *Resolver) Managers(args struct{ UserId graphql.ID }) ([]*contactResolver, error) {
	userReq := &contacts.UserRequest{UserId: string(args.UserId)}
	ctx := context.Background()
	contactsService, err := getContactsServiceClient()
	if err != nil {
		logrus.Fatalf("unable to get contacts service client due to: %s", err.Error())
		return nil, err
	}
	var managers []*contactResolver
	person, err := contactsService.GetPerson(ctx, userReq)
	if err != nil {
		logrus.Errorf("failed due to contacts service error: %s", err.Error())
		return nil, err
	}
	pUid := person.Person.UserId
	reachedTop := false
	for !reachedTop {
		p, err := contactsService.GetManager(ctx, &contacts.UserRequest{UserId: pUid})
		if err != nil {
			logrus.Errorf("failed due to contacts service error: %s", err.Error())
			return managers, err
		}
		pUid = p.Person.UserId
		if p.Person == nil || p.Person.Id == 0 {
			reachedTop = true
			break
		}
		managers = append(managers,  &contactResolver{person: p.Person, err: err})
	}

	return managers, nil
}

// 4. Finally, implement the boilerplate: for each attribute exposed for the type (eg contact), in schema.go,
// implement a method that has a camel-cased name into the resolver.
// Implementation decision: evaluate if a field needs to be lazy-loaded (in these boilerplate attribute accessors), or
// eager-loaded in the API methods above. Experiment with various anticipated graphql queries to manage the
// trade-offs.
func (r *contactResolver) UserId() *string {
	if r.person != nil {
		return &r.person.UserId
	}
	return nil
}

func (r *contactResolver) Email() *string {
	if r.person != nil {
		return &r.person.Email
	}
	return nil
}

func (r *contactResolver) FirstName() *string {
	if r.person != nil {
		return &r.person.FirstName
	}
	return nil
}

func (r *contactResolver) LastName() *string {
	if r.person != nil {
		return &r.person.LastName
	}
	return nil
}

func (r *contactResolver) BuName() *string {
	if r.person != nil {
		return &r.person.Team.Org.Bu.Name
	}
	return nil
}

func (r *contactResolver) OrgName() *string {
	if r.person != nil {
		return &r.person.Team.Org.Name
	}
	return nil
}

func (r *contactResolver) TeamName() *string {
	if r.person != nil {
		return &r.person.Team.Name
	}
	return nil
}

func (r *contactResolver) ManagerUserId() *string {
	if r.person != nil {

		return &r.person.Manager.UserId
	}
	return nil
}

func (r *contactResolver) ManagerPreferredName() *string {
	if r.person != nil {

		return &r.person.Manager.PreferredName
	}
	return nil
}

func (r *contactResolver) Title() *string {
	if r.person != nil {
		return &r.person.Title
	}
	return nil
}

func (r *contactResolver) TwitchAmznUserId() *string {
	if r.person != nil {
		return &r.person.TwitchAmznUid
	}
	return nil
}

func (r *contactResolver) EmployeeNum() *graphql.ID {
	if r.person != nil {
		id := graphql.ID(int(r.person.EmployeeNumber))
		return &id
	}
	return nil
}

func (r *contactResolver) PreferredName() *string {
	if r.person != nil {
		return &r.person.PreferredName
	}
	return nil
}
