package apiv2

import (
	graphql "github.com/neelance/graphql-go"

	"code.justin.tv/availability/goracle/catalog"
)

// Queries
type featureResolver struct {
	f *catalog.Feature
}

func (r *Resolver) Feature(args struct{ ID graphql.ID }) (*featureResolver, error) {
	return resolveFeature(args.ID)
}

func (r *Resolver) Features(args struct {
	AttributeName  *string
	AttributeValue *string
}) ([]*featureResolver, error) {
	ids, err := idsWithAttribute(catalog.LogTypeFeature, args.AttributeName, args.AttributeValue)
	if err != nil {
		return nil, err
	}
	features, err := catalog.GetCatalog().GetFeaturesByIDs(ids)
	if err != nil {
		return nil, err
	}
	featureResolvers := []*featureResolver{}
	for _, feature := range features {
		featureResolvers = append(featureResolvers, &featureResolver{f: feature})
	}
	return featureResolvers, nil
}

func resolveFeature(id graphql.ID) (*featureResolver, error) {
	u, err := idStringToUint(id)
	if err != nil {
		return nil, err
	}
	if u == 0 {
		return nil, nil
	}
	feature, err := catalog.GetCatalog().GetFeatureByID(u)
	if err != nil {
		return nil, err
	}
	return &featureResolver{
		f: feature,
	}, nil
}

func (r *featureResolver) ID() graphql.ID {
	return idUintToString(r.f.ID)
}

func (r *featureResolver) Name() string {
	return r.f.Name
}

func (r *featureResolver) Label() string {
	return r.f.Label
}

func (r *featureResolver) Description() string {
	return r.f.Description
}

func (r *featureResolver) Metrics() ([]*metricResolver, error) {
	metrics := []*metricResolver{}
	for _, metric := range r.f.Metrics {
		metrics = append(metrics, &metricResolver{m: metric})
	}
	return metrics, nil
}

func (r *featureResolver) MetricIDs() ([]graphql.ID, error) {
	metricIDs := []graphql.ID{}
	for _, metric := range r.f.Metrics {
		metricIDs = append(metricIDs, idUintToString(metric.ID))
	}
	return metricIDs, nil
}

func (r *featureResolver) Parent() (*featureResolver, error) {
	// For some reason, features use *uint instead of uint for their FKs
	// which is different than every other model we have, which means
	// you compare to nil instead of compare to 0
	if r.f.FeatureID != nil {
		id := idUintToString(*r.f.FeatureID)
		return resolveFeature(id)
	}
	return nil, nil
}

func (r *featureResolver) ParentID() *graphql.ID {
	if r.f.FeatureID != nil {
		id := idUintToString(*r.f.FeatureID)
		return &id
	}
	return nil
}

func (r *featureResolver) Children() ([]*featureResolver, error) {
	features := []*featureResolver{}
	for _, feature := range r.f.Children {
		features = append(features, &featureResolver{f: feature})
	}
	return features, nil
}

func (r *featureResolver) ChildrenIDs() ([]graphql.ID, error) {
	featureIDs := []graphql.ID{}
	for _, feature := range r.f.Children {
		featureIDs = append(featureIDs, idUintToString(feature.ID))
	}
	return featureIDs, nil
}

func (r *featureResolver) Attributes() []*attributeResolver {
	res, err := resolveAttributes(catalog.LogTypeFeature, r.f.ID)
	if err != nil {
		return nil
	}
	return res
}
