package apiv2

import (
	"bytes"
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	goji "goji.io"

	"goji.io/pat"

	graphql "github.com/neelance/graphql-go"
	"github.com/neelance/graphql-go/relay"
	"github.com/sirupsen/logrus"
	"github.com/stretchr/testify/assert"

	"code.justin.tv/availability/goracle/catalog"
)

func init() {
	dsn := "sqlite3://file::memory:?mode=memory&cache=shared"
	// dsn = "sqlite3:///tmp/goracle.sqlite"
	catalog.SetupDB(dsn)
	catalog.AddBaseServiceTypes()
}

type GraphqlQuery struct {
	Operation *string                 `json:"operation"`
	Query     *string                 `json:"query"`
	Variables *map[string]interface{} `json:"variables"`
}

func simpleQuery(q string) string {
	g := GraphqlQuery{
		Query: &q,
	}
	b, err := json.Marshal(g)
	if err != nil {
		logrus.Fatal(err)
	}
	return string(b)
}

var graphqlTests = []struct {
	name string
	in   string
	out  string
}{
	{
		name: "create attribute",
		in:   simpleQuery(`mutation {createAttribute(attribute: {object_id:"1", object_type: "attribute", name: "tier", value:"1"}) {id, name, value}}`),
		out:  `{"data":{"createAttribute":{"id":"1", "name":"tier", "value":"1"}}}`,
	},
	{
		name: "create service",
		in:   simpleQuery(`mutation {createService(service: {name: "sname1", primary_owner_id:"100324286", state:"active", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id}}`),
		out:  `{"data":{"createService":{"id":"1", "name": "sname1", "primary_owner_id":"100324286"}}}`,
	},
	{
		name: "get service",
		in:   simpleQuery(`{service(id: "1") {id, name, pagerduty, slack}}`),
		out:  `{"data":{"service":{"id": "1", "name":"sname1", "pagerduty":"pagerduty2", "slack": "svcslack"}}}`,
	},
	{
		name: "get logs with offset 1 limit 2 (starting from most recent)",
		in:   simpleQuery(`{logs(offset:1,limit:2) {id,action,item_type,item_label,item_id,author}}`),
		out:  `{"data":{"logs":[{"id":"2", "action":"update", "item_type":"attribute", "item_label":"tier", "item_id":"1", "author":""}, {"item_type":"attribute", "item_label":"tier", "item_id":"1", "author":"", "id":"1", "action":"create"}]}}`,
	},
	{
		name: "get single log for particular item",
		in:   simpleQuery(`{logs(item_type: "service", item_id: "1", limit:1) {id,action,item_type,item_label,item_id,author}}`),
		out:  `{"data":{"logs":[{"id":"3", "action":"create", "item_type":"service", "item_label":"sname1", "item_id":"1", "author":""}]}}`,
	},
	{
		name: "get logs with action filter",
		in:   simpleQuery(`{logs(action: "create", limit:2) {id,action,item_type,item_label,item_id,author}}`),
		out:  `{"data":{"logs":[{"id":"3", "action":"create", "item_type":"service", "item_label":"sname1", "item_id":"1", "author":""},{"item_type":"attribute", "item_label":"tier", "item_id":"1", "author":"", "id":"1", "action":"create"}]}}`,
	},
	{
		name: "create service2",
		in:   simpleQuery(`mutation {createService(service: {name: "sname2", primary_owner_id: "100324286", state:"active", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id}}`),
		out:  `{"data":{"createService":{"id":"2","name":"sname2","primary_owner_id":"100324286"}}}`,
	},
	{
		name: "create service3",
		in:   simpleQuery(`mutation {createService(service: {name: "sname3", primary_owner_id: "100324286", state:"active", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id}}`),
		out:  `{"data":{"createService":{"id":"3","name":"sname3","primary_owner_id":"100324286"}}}`,
	},
	{
		name: "create service dependency",
		in:   simpleQuery(`mutation {createServiceDependency(service_dependency: {root_service_id: "1", downstream_service_id: "2"}) {id, root_service_id, downstream_service_id}}`),
		out:  `{"data":{"createServiceDependency":{"id":"1", "root_service_id":"1", "downstream_service_id": "2"}}}`,
	},
	{
		name: "create service dependency2",
		in:   simpleQuery(`mutation {createServiceDependency(service_dependency: {root_service_id: "1", downstream_service_id: "3"}) {root_service_id, downstream_service_id}}`),
		out:  `{"data":{"createServiceDependency":{"root_service_id":"1", "downstream_service_id": "3"}}}`,
	},
	{
		name: "update service upstreams from service ((test keep dep))",
		in:   simpleQuery(`mutation {updateService(id:"2",  service: {primary_owner_id: "100324286", type: "internal-product", service_upstreams:[{id:"1", root_service_id: "1", downstream_service_id: "2"},{root_service_id:"3", downstream_service_id:"2"}], attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {service_upstreams {root_service_id, downstream_service_id}}}`),
		out:  `{"data":{"updateService":{"service_upstreams": [{"root_service_id": "1", "downstream_service_id": "2"},{"root_service_id":"3", "downstream_service_id":"2"}]}}}`,
	},
	{
		name: "update service downstreams from service (test remove dep)",
		in:   simpleQuery(`mutation {updateService(id:"1", service: {primary_owner_id: "100324286",type: "internal-product", service_downstreams:[{root_service_id:"1", downstream_service_id:"2"}], attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {service_downstreams {root_service_id, downstream_service_id}}}`),
		out:  `{"data":{"updateService":{"service_downstreams": [{"root_service_id":"1", "downstream_service_id":"2"}]}}}`,
	},
	{
		name: "update service upstreams from service (using serviceInput)",
		in:   simpleQuery(`mutation {updateService(id:"1", service: {primary_owner_id: "100324286", type: "internal-product", service_upstreams:[{root_service:{id:"3"}, downstream_service:{id:"1"}}], attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {service_upstreams {root_service_id, downstream_service_id}}}`),
		out:  `{"data":{"updateService":{"service_upstreams": [{"root_service_id":"3", "downstream_service_id":"1"}]}}}`,
	},
	{
		name: "create service with no name",
		in:   simpleQuery(`mutation {createService(service: {pagerduty: "pagerduty2", primary_owner_id: "100324286", state:"active",  slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name}}`),
		out:  `{"data":{"createService":null}, "errors": [{"message": "service name cannot be empty", "path": ["createService"]}]}`,
	},
	{
		name: "create service with no svctype",
		in:   simpleQuery(`mutation {createService(service: {name: "nosvctype", primary_owner_id: "100324286", state:"active",  pagerduty: "pagerduty2", slack: "svcslack", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name}}`),
		out:  `{"data":{"createService":null}, "errors": [{"message": "missing required field: 'serviceType'", "path": ["createService"]}]}`,
	},
	{
		name: "create service with unresolvable owner",
		in:   simpleQuery(`mutation {createService(service: {name: "snamenoowner", state:"active", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}], primary_owner_id:"9999999"}) {id,name}}`),
		out:  `{"data":{"createService":null},"errors":[{"message":"Could not resolve given primary_owner_id: user with id '9999999' was not found in LDAP","path":["createService"]}]}`,
	},
	{
		name: "create service with same name",
		in:   simpleQuery(`mutation {createService(service: {name: "sname1", state:"active", pagerduty: "pagerduty2", primary_owner_id: "100324286", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name}}`),
		out:  `{"data":{"createService":null}, "errors": [{"message": "UNIQUE constraint failed: services.name", "path": ["createService"]}]}`,
	},
	{
		name: "create service without tier attribute",
		in:   simpleQuery(`mutation {createService(service: {name: "sname1", primary_owner_id: "100324286", state:"active", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product"}) {id,name}}`),
		out:  `{"data":{"createService":null}, "errors": [{"message": "missing required field: attribute 'tier'", "path": ["createService"]}]}`,
	},
	{
		name: "update service name but don't provide primary owner",
		in:   simpleQuery(`mutation {updateService(id:"2", service: {name: "updateservicename", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name}}`),
		out:  `{"data": {"updateService":{"name":"updateservicename", "id": "2"}}}`,
	},
	{
		name: "update service name",
		in:   simpleQuery(`mutation {updateService(id:"2", service: {name: "updateservicename", primary_owner_id: "100324286", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name}}`),
		out:  `{"data":{"updateService":{"id":"2","name":"updateservicename"}}}`,
	},
	{
		name: "update service type to empty",
		in:   simpleQuery(`mutation {updateService(id:"2", service: {name: "servicetypeempty", pagerduty: "pagerduty2", slack: "svcslack", type:"", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,type}}`),
		out:  `{"data":{"updateService":null},"errors":[{"message":"no such service type ''","path":["updateService"]}]}`,
	},
	{
		name: "update service unresolvable primary_owner_id",
		in:   simpleQuery(`mutation {updateService(id:"2", service: {name: "updateunresolvedprimaryownerid", primary_owner_id: "99999999", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name}}`),
		out:  `{"data":{"updateService":null},"errors":[{"message":"Could not resolve given primary_owner_id: user with id '99999999' was not found in LDAP","path":["updateService"]}]}`,
	},
	{
		name: "update service primary_owner_id=0",
		in:   simpleQuery(`mutation {updateService(id:"2", service: {name: "updateprimaryownerid0", primary_owner_id: "0", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name}}`),
		out:  `{"data":{"updateService":null},"errors":[{"message":"Cannot set Primary Owner to empty!","path":["updateService"]}]}`,
	},
	{
		name: "update service primary_owner_uid=0",
		in:   simpleQuery(`mutation {updateService(id:"2", service: {name: "updateprimaryowneruid0", primary_owner_uid: "0", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name}}`),
		out:  `{"data":{"updateService":null},"errors":[{"message":"Cannot set Primary Owner to empty!","path":["updateService"]}]}`,
	},
	{// needs a working context from guardianauth.GetUserFromContext(ctx)
		name: "update service, different primary_owner_id",
		in:   simpleQuery(`mutation {updateService(id:"2", service: {name: "updatevalidprimaryownerid", primary_owner_id: "100324286", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id}}`),
		out:  `{"data":{"updateService":{"id":"2","name":"updatevalidprimaryownerid","primary_owner_id":"100324286"}}}`,
	},
	{// needs a working context from guardianauth.GetUserFromContext(ctx)
		name: "update service, different primary_owner_uid",
		in: simpleQuery(`mutation {updateService(id:"2", service: {name: "updatevalidprimaryowneruid", primary_owner_uid: "emmett", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id}}`),
		out: `{"data":{"updateService":{"id":"2","name":"updatevalidprimaryowneruid","primary_owner_id":"100324286"}}}`,
	},
	{// needs a working context from guardianauth.GetUserFromContext(ctx)
		name: "update service, no tier attribute",
		in:   simpleQuery(`mutation {updateService(id:"2", service: {name: "updatewithpreexistingtier", primary_owner_id: "100324286", pagerduty: "pagerduty2", slack: "svcslack", type: "internal-product"}) {id,name,primary_owner_id}}`),
		out:  `{"data":{"updateService":{"id":"2","name":"updatewithpreexistingtier","primary_owner_id":"100324286"}}}`,
	},
	// -- DISABLE THESE UNTIL JENKINS CAN INVOKE FULTONIZED CONTACTS SERVICE -- //
	//{
	//	name: "query contact info of single specified uid",
	//	in:   simpleQuery(`{contacts(user_ids:["emmett"]) { user_id,email,manager_user_id,bu_name,org_name,team_name,manager_preferred_name}}`),
	//	out:  `{"data":{"contacts":[{"user_id":"emmett","email":"emmett@twitch.tv","manager_user_id":"","bu_name":"","org_name":"","team_name":"Executive Suite","manager_preferred_name":""}]}}`,
	//},
	//{
	//	name: "query contact info of single bad uid",
	//	in:   simpleQuery(`{contacts(user_ids:["aljksdfjlk"]) { user_id,email,manager_user_id,bu_name,org_name,team_name}}`),
	//	out:  `{"data":{"contacts":[]}}`,
	//},
	//{
	//	name: "query contact info without passing any uid",
	//	in:   simpleQuery(`{contacts(user_ids:[""]) { user_id,email,manager_user_id,bu_name,org_name,team_name}}`),
	//	out:  `{"data":{"contacts":[]}}`,
	//},
	//{
	//	name: "query contact info of single specified uid with valid manager",
	//	in:   simpleQuery(`{contacts(user_ids:["beardenj"]) { user_id,email,manager_user_id,bu_name,org_name,team_name,manager_preferred_name}}`),
	//	out:  `{"data":{"contacts":[{"bu_name":"Platform & Services", "org_name":"Ops Excellence", "team_name":"OE Productivity Engineering", "manager_preferred_name":"Tyler Robbins", "user_id":"beardenj", "email":"beardenj@twitch.tv", "manager_user_id":"tylerobb"}]}}`,
	//},
	//{
	//	name: "query managers' contact info of single specified uid",
	//	in:   simpleQuery(`{managers(user_id:"lukemng") { user_id }}`),
	//	out:  `{"data":{"managers":[{"user_id":"beardenj"},{"user_id":"tylerobb"},{"user_id":"ethan"},{"user_id":"emmett"}]}}`,
	//},
	{
		name: "create service without state field",
		in:   simpleQuery(`mutation {createService(service: {name: "service-no-state", primary_owner_id:"100324286", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id}}`),
		out:  `{"data":{"createService":null}, "errors": [{"message": "missing required field: 'state'", "path": ["createService"]}]}`,
	},
	{
		name: "create service in active state",
		in:   simpleQuery(`mutation {createService(service: {name: "service-active", primary_owner_id:"100324286", state: "aCtIVe", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id,state}}`),
		out:  `{"data":{"createService":{"name":"service-active", "primary_owner_id":"100324286", "state":"Active", "id":"4"}}}`,
	},
	{
		name: "create service in inactive state",
		in:   simpleQuery(`mutation {createService(service: {name: "service-inactive", primary_owner_id:"100324286", state: "iNaCtIVe", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id,state}}`),
		out:  `{"data":{"createService":{"name":"service-inactive", "primary_owner_id":"100324286", "state":"Inactive", "id":"5"}}}`,
	},
	{
		name: "create service in invalid state",
		in:   simpleQuery(`mutation {createService(service: {name: "service-foobar-state", primary_owner_id:"100324286", state: "foobar123", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,name,primary_owner_id,state}}`),
		out:  `{"data":{"createService":null},"errors":[{"message":"no such service state 'foobar123'","path":["createService"]}]}`,
	},
	{
		name: "update service state active to inactive",
		in:   simpleQuery(`mutation {updateService(id:"4", service: {state: "Inactive", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,state}}`),
		out:  `{"data":{"updateService":{"id":"4","state":"Inactive"}}}`,
	},
	{
		name: "update service state inactive to active",
		in:   simpleQuery(`mutation {updateService(id:"4", service: {state: "Active", type: "internal-product", attributes:[{name: "tier", value:"1", id:"1", object_id:"1", object_type:"attribute"}]}) {id,state}}`),
		out:  `{"data":{"updateService":{"id":"4","state":"Active"}}}`,
	},
}

func TestGraphQLAPIIntegration(t *testing.T) {
	mux := goji.NewMux()
	schema, err := graphql.ParseSchema(GetSchema(), &Resolver{})
	if err != nil {
		panic(err)
	}
	urlpath := "/api/v2/query"
	mux.Handle(pat.New(urlpath), &relay.Handler{Schema: schema})
	for _, tt := range graphqlTests {
		t.Run(tt.name, func(t *testing.T) {
			if testing.Verbose() {
				t.Logf("Starting test case %s", tt.name)
				t.Logf("    Input: %s", tt.in)
				t.Logf("Expecting: %s", tt.out)
			}
			req, err := http.NewRequest("POST", urlpath, bytes.NewBufferString(tt.in))
			if !assert.NoError(t, err, "NewRequest fail for: %s", tt.name) {
				t.Errorf("NewRequest fail for: %s", tt.name)
			}
			w := httptest.NewRecorder()
			mux.ServeHTTP(w, req)
			if !assert.Equal(t, http.StatusOK, w.Code, "Wrong HTTP status for: %s", tt.name) {
				t.Errorf("Wrong HTTP status for: %s", tt.name)
			}
			bod := w.Body.Bytes()
			if testing.Verbose() {
				t.Logf("   Output: %s", bod)
			}
			if !assert.JSONEq(t, tt.out, string(bod), "Returned JSON doesn't match for: %s", tt.name) {
				t.Errorf("Returned JSON doesn't match for: %s", tt.name)
			}
		})
	}
}
