package apiv2

import (
	"context"
	"net/http"

	"code.justin.tv/qe/twitchldap"
	"code.justin.tv/sse/malachai/pkg/events"
	"code.justin.tv/sse/malachai/pkg/s2s/callee"
	graphql "github.com/neelance/graphql-go"
	"github.com/neelance/graphql-go/relay"
	"github.com/rs/cors"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"

	"bytes"
	"code.justin.tv/availability/goracle/config"
	"code.justin.tv/availability/goracle/goracleUser"
	"code.justin.tv/availability/goracle/guardianauth"
	"code.justin.tv/availability/goracle/stats"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"strings"
	"regexp"
)

// This page is for interacting with graphql in a local dev
// environment
var page = []byte(`
<!DOCTYPE html>
<html>
	<head>
		<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/graphiql/0.10.2/graphiql.css" />
		<script src="https://cdnjs.cloudflare.com/ajax/libs/fetch/1.1.0/fetch.min.js"></script>
		<script src="https://cdnjs.cloudflare.com/ajax/libs/react/15.5.4/react.min.js"></script>
		<script src="https://cdnjs.cloudflare.com/ajax/libs/react/15.5.4/react-dom.min.js"></script>
		<script src="https://cdnjs.cloudflare.com/ajax/libs/graphiql/0.10.2/graphiql.js"></script>
	</head>
	<body style="width: 100%; height: 100%; margin: 0; overflow: hidden;">
		<div id="graphiql" style="height: 100vh;">Loading...</div>
		<script>
			function graphQLFetcher(graphQLParams) {
				return fetch("/api/v2/query.guardian", {
					method: "post",
					body: JSON.stringify(graphQLParams),
					credentials: "include",
				}).then(function (response) {
					return response.text();
				}).then(function (responseBody) {
					try {
						return JSON.parse(responseBody);
					} catch (error) {
						return responseBody;
					}
				});
			}
			ReactDOM.render(
				React.createElement(GraphiQL, {fetcher: graphQLFetcher}),
				document.getElementById("graphiql")
			);
		</script>
	</body>
</html>
`)

func RegisterHandlers(mux *goji.Mux) {
	subMux := goji.SubMux()
	c := cors.New(cors.Options{
		AllowedOrigins: []string{"http://localhost:3000", "http://servicecatalog-local.internal.justin.tv:3000", "https://servicecatalog-dev.internal.justin.tv", "https://servicecatalog.internal.justin.tv"},
		AllowedMethods: []string{"GET", "POST", "HEAD", "OPTIONS", "PUT", "DELETE", "PATCH"},
		AllowedHeaders:   []string{"*"},
		MaxAge: 86400,
		AllowCredentials: true,
	})
	// register same old middlewares
	subMux.Use(c.Handler)
	mux.Handle(pat.New("/api/v2/*"), subMux)
	// Order is important here because unprotected endpoint has api/v2/* mux
	registerGuardianEndpoint(subMux)
	if config.Config.EnableS2S {
		registerS2SEndpoint(subMux)
	}
	registerUnprotectedEndpoint(subMux)
}

type graphqlRequest struct {
	OperationName string                 `json:"operationName"`
	Query         string                 `json:"query"`
	Variables     map[string]interface{} `json:"variables"`
}

// regexp compilation is rather expensive, so maintain a cache
var regexCache = map[string]*regexp.Regexp{
	"mutation": regexp.MustCompile("mutation.*{"),  // pre-populate with a regex to catch non-readonly queries
}

// readOnlyWrapper logs mutation queries, and optionally blocks mutation queries if the referer header matches specified pattern
func readOnlyWrapper(h http.Handler, hardReadOnlyReferrerPattern string) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		body, err := ioutil.ReadAll(r.Body)
		// Restore the io.ReadCloser to its original state so the wrapped handler can use it.
		r.Body = ioutil.NopCloser(bytes.NewBuffer(body))
		if err != nil {
			logrus.Warnf("Unable to parse request body in gql readonly check %s", err.Error())
		}
		var gr graphqlRequest
		err = json.Unmarshal(body, &gr)
		if err != nil {
			logrus.Warnf("Unable to parse from json request body in gql readonly check %s", err.Error())
		}
		if gr.Query != "" {
			trimmedQuery := strings.TrimSpace(gr.Query)
			if regexCache["mutation"].MatchString(trimmedQuery) {
				// soft enforce - legacy behavior - log a warning - caller (client) is not informed
				logrus.Warn("DEPRECATED: Mutation query made over the query readonly endpoint")
				logrus.Warnf("Source IP is: %v", r.RemoteAddr)
				logrus.Warnf("Source Header is: %v", r.Header)
				logrus.Warnf("Operation is: %s", gr.OperationName)
				logrus.Warnf("Query is: %s", trimmedQuery)
				logrus.Warnf("Variables is: %v", gr.Variables)
				// hard enforce - actually return an error to caller (client)
				if _, cacheHit := regexCache[hardReadOnlyReferrerPattern]; !cacheHit {
					regexCache[hardReadOnlyReferrerPattern] = regexp.MustCompile(hardReadOnlyReferrerPattern)
				}
				// enforce ReadOnly mode if it is set in config or based on url-matching
				enforceReadOnly := config.IsReadOnlyMode() || regexCache[hardReadOnlyReferrerPattern].MatchString(r.Header.Get("Referer"))
				if enforceReadOnly {
					w.WriteHeader(http.StatusForbidden)
					w.Write(bytes.NewBufferString("{\"error\": \"Mutation query is forbidden\"}").Bytes())
					return
				}
			}
		}
		h.ServeHTTP(w, r)
	})
}

func registerUnprotectedEndpoint(apiMux *goji.Mux) {
	unprotected := goji.SubMux()
	apiMux.Handle(pat.New("/*"), unprotected)
	unprotected.Use(guardianauth.GuardianUser)

	// Add availabilty CSV report
	unprotected.Handle(pat.New("/report"), http.HandlerFunc(TeamAvailabilityReportHandler))

	// parse the defined query schema
	// also inject a stats tracer to track request timings and success/failures
	schema := graphql.MustParseSchema(GetSchema(), &Resolver{}, graphql.Tracer(stats.Tracer{Prefix: "noAuth"}))
	// register the schema to the query endpoint. blocks any caller that issued mutation query
	unprotected.Handle(pat.New("/query"), readOnlyWrapper(&relay.Handler{Schema: schema}, ".*"))
}

func registerGuardianEndpoint(apiMux *goji.Mux) {
	// parse the defined query schema
	// also inject a stats tracer to track request timings and success/failures
	//TODO set tracer to take in special flags
	schema := graphql.MustParseSchema(GetSchema(), &Resolver{}, graphql.Tracer(stats.Tracer{
		Prefix:   "guardianAuth",
		LogQuery: config.Config.EnableGraphqlLog,
	}))
	// register the schema to the query endpoint
	guardianWrapper := guardianauth.GuardianValidation(&relay.Handler{Schema: schema})
	// blocks dev console (/api/v2/dev) from running mutation query
	apiMux.Handle(pat.New("/query.guardian"), readOnlyWrapper(guardianWrapper, ".*/api/v2/dev$"))
	// serve dev console page
	devConsolePage := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.Write(page)
	})
	apiMux.Handle(pat.New("/dev"), guardianauth.GuardianValidation(devConsolePage))
}

func injectUserFromS2SCall(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		uid := r.Header.Get("authUser")
		if uid == "" {
			w.WriteHeader(http.StatusUnauthorized)
			w.Write([]byte("401 - Must provide valid LDAP username as header in authUser"))
			return
		}
		client, err := twitchldap.NewClient()
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
			w.Write([]byte(fmt.Sprintf("500 - encountered error when initiailizing twitchldap %s", err.Error())))
			return
		}
		user, err := client.GetUserInfoByName(uid)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
			w.Write([]byte(fmt.Sprintf("500 - encountered error when searching for user %s", err.Error())))
			return
		}
		if user == nil {
			w.WriteHeader(http.StatusUnauthorized)
			w.Write([]byte(fmt.Sprintf("401 - Could not find given user in LDAP %s", uid)))
			return
		}
		goracleUserInstance := &goracleUser.GoracleUser{
			UID:            user.UID,
			CN:             user.CN,
			EmployeeNumber: user.EmployeeNumber,
		}
		ctx := context.WithValue(r.Context(), goracleUser.GoracleUserKey, goracleUserInstance)
		h.ServeHTTP(w, r.WithContext(ctx))
	})
}

func registerS2SEndpoint(apiMux *goji.Mux) {
	logger := logrus.New()
	// Uncomment the below to get debug messages for s2s in the s2s directory (which you have to make in the root directory)
	//eventsWriterClient, err := fileeventbuffer.New(fileeventbuffer.Config{
	//	// This refers to s2s production
	//	Environment: "production",
	//	LogDir:      "s2s",
	//}, logger)
	eventsWriterClient, err := events.NewEventLogger(events.Config{
		Environment: "production",
	}, logger)
	if err != nil {
		logrus.Fatal("Could not initialize s2s event writer: ", err.Error())
	}
	s2sMiddleware := &callee.Client{
		ServiceName:        config.S2sService(),
		Logger:             logger,
		EventsWriterClient: eventsWriterClient,
	}
	err = s2sMiddleware.Start()
	if err != nil {
		logrus.Errorf("Could not start s2s middleware: ", err.Error())
		return
	}
	// parse the defined query schema
	// also inject a stats tracer to track request timings and success/failures
	schema := graphql.MustParseSchema(GetSchema(), &Resolver{}, graphql.Tracer(stats.Tracer{
		Prefix:   "s2sAuth",
		LogQuery: config.Config.EnableGraphqlLog}))
	apiMux.Handle(pat.New("/query.s2s"), s2sMiddleware.CapabilitiesInjectorMiddleware(injectUserFromS2SCall(&relay.Handler{Schema: schema})))
}
